from typing import List

import numpy as np
import torch
import torch.nn.functional as F
from torch import Tensor

from base_model import BaseRGBDModel
from device import device, is_cuda_available
from s_multimae.checkpoint import load_checkpoint_for_inference
from s_multimae.configs.base_config import base_cfg
from s_multimae.data_augmentation import get_data_augmentation
from s_multimae.rgbd_model import RGBDModel
from s_multimae.run_type import RUN_TYPE


class RGBDMultiMAEModel(BaseRGBDModel):
    def __init__(self, cfg: base_cfg):
        """Wrapper of RGBDModel"""
        super(RGBDMultiMAEModel, self).__init__()
        self.model = RGBDModel(cfg, run_type=RUN_TYPE.HUGGINGFACE)
        self.cfg = cfg

        load_checkpoint_for_inference(
            self.model, 
            cfg.ckpt_path
        )
        self.model.to(device)
        self.model.eval()

        self.data_augmentation = get_data_augmentation(
            cfg, cfg.test_image_size, cfg.is_padding_for_test
        )

    def inference(
        self, image: Tensor, depth: Tensor,
    ) -> np.ndarray:
        origin_shape = image.shape

        # 1. Preprocessing
        images = image.unsqueeze(0)
        depths = depth.unsqueeze(0)

        images, depths = self.data_augmentation.preprocessing(images, depths)

        # 2. Inference
        images, depths = images.to(device), depths.to(device)
        with torch.autocast(device.type, enabled=is_cuda_available):
            pred_no_sigmoid = self.model.inference(images, depths)
        
        # 3. Return saliency maps
        res: Tensor = F.interpolate(
            pred_no_sigmoid, size=(origin_shape[1], origin_shape[2]), 
            mode='bilinear', align_corners=False
        )
        res = res.sigmoid().squeeze().data.cpu().numpy()
        # res = normalize(res)

        if is_cuda_available:
            res = np.float32(res)

        return res

    def batch_inference(
        self, images: Tensor, depths: Tensor,
    ) -> List[np.ndarray]:
        origin_shape = images.shape

        # 1. Preprocessing
        images, depths = self.data_augmentation.preprocessing(images, depths)

        # 2. Inference
        images, depths = images.to(device), depths.to(device)
        with torch.autocast(device.type, enabled=is_cuda_available):
            pred_no_sigmoid = self.model.inference(images, depths)
        
        # 3. Return saliency maps
        res: Tensor = F.interpolate(
            pred_no_sigmoid, size=(origin_shape[2], origin_shape[3]), 
            mode='bilinear', align_corners=False
        )
        res = res.sigmoid().squeeze(1).data.cpu().numpy()
        # res = [normalize(e) for e in res]

        if is_cuda_available:
            res = [np.float32(e) for e in res]

        return res