import os
from typing import List, Optional, Tuple, Union

import torch

from app_utils import get_size, normalize
from base_model import BaseRGBDModel
from depth_model import BaseDepthModel

# from transformers import AutoModel

TORCH_VERSION = ".".join(torch.__version__.split(".")[:2])
CUDA_VERSION = torch.__version__.split("+")[-1]
print("torch: ", TORCH_VERSION, "; cuda: ", CUDA_VERSION)

import cv2
# Imports
import numpy as np
import torch
import torch.nn.functional as F
import torchvision.transforms.functional as TF
from PIL import Image
from torch import Tensor, nn

from visualizer import VisImage, Visualizer

# Environment
torch.set_grad_enabled(False)
from device import device

print(f'device: {device}')

# TODO Not working!
# access_token = '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'
# sod_model: multimae.RGBDModel = AutoModel.from_pretrained(
#     "HCMUT-LVTN-Thinh/rgbdsod-multimae-model", trust_remote_code=True,
#     use_auth_token=access_token
# )

def predict_sod(
    sod_model: BaseRGBDModel,
    image: Image.Image, 
    depth: Tensor, 
    visualizer: Visualizer,
    color: np.ndarray = None,
) -> np.ndarray:
    res = sod_model.inference(image, depth)
    res[res < 0.5] = 0.0
    res[res >= 0.5] = 1.0

    vis_image: VisImage = visualizer.draw_binary_mask(res, color)
    return vis_image.get_image()[:, :, ::-1]

def post_processing_depth(depth: np.ndarray) -> np.ndarray:
    depth = (normalize(depth) * 255).astype(np.uint8)
    return cv2.applyColorMap(depth, cv2.COLORMAP_OCEAN)

def base_inference(
    depth_model: BaseDepthModel,
    sod_model: BaseRGBDModel,
    raw_image: Union[Image.Image, np.ndarray], 
    raw_depth: Optional[Union[Image.Image, np.ndarray]] = None, 
    color: np.ndarray = None
) -> Tuple[np.ndarray, np.ndarray, np.ndarray]:
    """Inference a pair of rgb image and depth image
    if depth image is not provided, the depth_model will predict a depth image based on image
    """
    origin_size = get_size(raw_image)

    image = TF.to_tensor(raw_image)

    # Predict depth
    if raw_depth is None:
        depth: Tensor = depth_model.forward(image)
    else:
        depth = TF.to_tensor(raw_depth)

    sm = sod_model.inference(image, depth)
    binary_mask = np.array(sm)
    binary_mask[binary_mask < 0.5] = 0.0
    binary_mask[binary_mask >= 0.5] = 1.0

    visualizer = Visualizer(raw_image)
    vis_image: VisImage = visualizer.draw_binary_mask(binary_mask, color)
    sod = vis_image.get_image()[:, :, ::-1]

    depth = depth.permute(1,2,0).detach().cpu().numpy()
    depth = cv2.resize(depth, origin_size)
    depth = post_processing_depth(depth)

    return depth, sod, sm

def transform_images(inputs: List[Image.Image], transform: nn.Module) -> Tensor:
    if len(inputs) == 1:
        return transform(inputs[0]).unsqueeze(0)
    return torch.cat([transform(input).unsqueeze(0) for input in inputs])

def batch_base_inference(
    depth_model: BaseDepthModel,
    sod_model: BaseRGBDModel,
    raw_images: List[Union[Image.Image, np.ndarray]], 
    color: np.ndarray = None
) -> Tuple[List[np.ndarray], List[np.ndarray]]:
    """Inference a batch of pairs of rgb image and depth image"""
    if len(raw_images) == 0:
        return [], []

    origin_size = get_size(raw_images[0])

    images = transform_images(raw_images, TF.to_tensor)
    depths: Tensor = depth_model.batch_forward(images)

    res = sod_model.batch_inference(images, depths)

    sods: List[np.ndarray] = []
    for i, e in enumerate(res):
        e[e < 0.5] = 0.0
        e[e >= 0.5] = 1.0

        visualizer = Visualizer(raw_images[i])
        vis_image: VisImage = visualizer.draw_binary_mask(e, color)
        sod = vis_image.get_image()[:, :, ::-1]
        sods.append(sod)

    depths = depths.permute(0,2,3,1).detach().cpu().numpy()
    depths = [post_processing_depth(cv2.resize(depth, origin_size)) for depth in depths]
    return depths, sods