import os

import torch
import torchvision.transforms as transforms
import torchvision.transforms.functional as TF
from torch import Tensor, nn

from app_utils import count_parameters
from device import device
from dpt.models import DPTDepthModel
from lib.multi_depth_model_woauxi import RelDepthModel
from lib.net_tools import load_ckpt


class BaseDepthModel:
    def __init__(self, image_size: int) -> None:
        self.image_size = image_size
        self.model: nn.Module = None

    def forward(self, image: Tensor)->Tensor:
        """Perform forward inference for an image
        Input image of shape [c, h, w]
        Return of shape [c, h, w]
        """
        raise NotImplementedError()

    def batch_forward(self, images: Tensor)->Tensor:
        """Perform forward inference for a batch of images
        Input images of shape [b, c, h, w]
        Return of shape [b, c, h, w]"""
        raise NotImplementedError()

    def get_number_of_parameters(self) -> int:
        return count_parameters(self.model)

class DPTDepth(BaseDepthModel):
    def __init__(self, image_size: int) -> None:
        super().__init__(image_size)
        print('DPTDepthconstructor')
        omnidata_ckpt = torch.load(
            os.path.join(
                'pretrained_models', 'rgb2depth', 
                'omnidata_rgb2depth_dpt_hybrid.pth'
            ),
            map_location='cpu'
        )
        self.model = DPTDepthModel()
        self.model.load_state_dict(omnidata_ckpt)
        self.model: DPTDepthModel = self.model.to(device).eval()

        self.transform = transforms.Compose([
            transforms.Resize(
                (self.image_size, self.image_size),
                interpolation=TF.InterpolationMode.BICUBIC
            ),
		    transforms.Normalize(
                (0.5, 0.5, 0.5), 
                (0.5, 0.5, 0.5), 
            )
        ])

    def forward(self, image: Tensor) -> Tensor:
        depth_model_input = self.transform(image.unsqueeze(0))
        return self.model.forward(depth_model_input.to(device)).squeeze(0)

    def batch_forward(self, images: Tensor)->Tensor:
        images: Tensor = TF.resize(
            images, (self.image_size, self.image_size), 
            interpolation=TF.InterpolationMode.BICUBIC
        )
        depth_model_input = (images - 0.5) / 0.5
        return self.model(depth_model_input.to(device))

class RelDepth(BaseDepthModel):
    def __init__(self, image_size: int) -> None:
        super().__init__(image_size)
        print('RelDepthconstructor')
        self.model: RelDepthModel = RelDepthModel(backbone='resnext101')
        load_ckpt(
            os.path.join(
                'pretrained_models', 
                'adelai_depth', 
                'res101.pth'
            ), 
            self.model
        )
        self.model = self.model.to(device).eval()
        self.transform = transforms.Compose([
            transforms.Resize(
                (448, 448),
                interpolation=TF.InterpolationMode.BICUBIC
            ),
		    transforms.Normalize(
                (0.485, 0.456, 0.406), 
                (0.229, 0.224, 0.225) 
            )
        ])

    def forward(self, image: Tensor) -> Tensor:
        images = self.transform(image.unsqueeze(0))
        pred_depth_ori = self.model.inference(images.to(device))
        pred_depth_ori = pred_depth_ori/pred_depth_ori.max()
        return pred_depth_ori.squeeze(0)

    def batch_forward(self, images: Tensor) -> Tensor:
        images: Tensor = TF.resize(
            images, (448, 448), 
            interpolation=TF.InterpolationMode.BICUBIC
        )
        images = self.transform(images)
        pred_depth_ori = self.model.inference(images.to(device))
        pred_depth_ori = pred_depth_ori/pred_depth_ori.max()
        return pred_depth_ori
        