import gradio as gr
import numpy as np
from PIL import Image
import tensorflow as tf
import tensorflow_hub as hub

style_transfer_model = hub.load("https://tfhub.dev/google/magenta/arbitrary-image-stylization-v1-256/2")

def perform_style_transfer(content_image, style_image):
    
    content_image = tf.convert_to_tensor(content_image, np.float32)[tf.newaxis, ...] / 255.
    style_image = tf.convert_to_tensor(style_image, np.float32)[tf.newaxis, ...] / 255.
    
    output = style_transfer_model(content_image, style_image)
    stylized_image = output[0]
    
    return Image.fromarray(np.uint8(stylized_image[0] * 255))


content_image_input = gr.inputs.Image(label="Content Image")
style_image_input = gr.inputs.Image(shape=(256, 256), label="Style Image")

# Examples
golden_gate = ["examples/golden_gate_bridge.jpeg", "examples/the_great_wave.jpeg"]
joshua_tree = ["examples/joshua_tree.jpeg", "examples/starry_night.jpeg"]
glacier = ["examples/glacier_national_park.jpeg", "examples/the_scream.jpg"]

app_interface = gr.Interface(fn=perform_style_transfer,
                             inputs=[content_image_input, style_image_input],
                             outputs="image",
                             title="Fast Neural Style Transfer",
                             description="""Gradio demo for Fast Neural Style Transfer using a pretrained Image Stylization model from TensorFlow Hub. To use it, simply upload a content image and style image, or click one of the examples to load them. To learn more about the project, please find the references listed below.<br>
                                         This Gradio Demo was build by <a href="https://huggingface.co/gstaff" target="_blank">Grant Stafford @gstaff</a>.""",
                             examples=[glacier, golden_gate, joshua_tree],
                             article="**References**\n\n"
                             "<a href='https://www.tensorflow.org/hub/tutorials/tf2_arbitrary_image_stylization' target='_blank'>1. Tutorial to implement Fast Neural Style Transfer using the pretrained model from TensorFlow Hub</a>  \n"
                             "<a href='https://huggingface.co/spaces/luca-martial/neural-style-transfer' target='_blank'>2. The idea to build a neural style transfer application was inspired from this Hugging Face Space </a>",
                             theme='gstaff/whiteboard')
app_interface.launch()
