import numpy as np
from glob import glob
import cv2
import gradio as gr
import torch
import torchvision.models as models
from PIL import Image
import torch.optim as optim
import torchvision.transforms as transforms
from torchvision import datasets
from PIL import ImageFile
import torch.nn as nn
from collections import OrderedDict
import pathlib

dog_files = np.array(glob('dogImages/*/*/*'))
human_files = np.array(glob('lfw/*/*'))

# Human face detector using OpenCV
def detector_humanface(image_path):
  # Loading color image (RGB)
  # image = cv2.imread(image_path)
  image = image_path
  # Converting color image to grayscale
  grayscale_img = cv2.cvtColor(np.array(image), cv2.COLOR_BGR2GRAY)
  # Taking pre-trained human face detector classifier
  face_cascade = cv2.CascadeClassifier(cv2.data.haarcascades + 'haarcascade_frontalface_alt.xml')
  # finding face in grayscale image
  faces = face_cascade.detectMultiScale(grayscale_img)
  return len(faces) > 0

# detecting dogs using pre-trained model
vggmodel = models.vgg16(pretrained=True)

# check if CUDA is available
use_cuda = torch.cuda.is_available()

# move model to GPU if CUDA is available
if use_cuda:
    vggmodel = vggmodel.cuda()

# detecting dog in an image
def vgg_model(img_path):
  # move model to GPU if CUDA is available
  use_cuda = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
  # image = Image.open(img_path)
  image = img_path
  transform = transforms.Compose([transforms.ToTensor()])
  tensor_image = transform(image)
  image_tranformation = transforms.Compose(
      [
       transforms.CenterCrop(224),
       transforms.ToTensor(),
       transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225]) # normalizing each channel RGB
      ]
  )

  # changing the dimension using unsqueeze & taking all tensors into size one list 
  transformed_img = image_tranformation(image).unsqueeze(0).to(use_cuda)
  detected_dog = vggmodel(transformed_img)
  _, pred = torch.max(detected_dog, 1)
  # return the index of predicted class
  return pred[0]

def dog_detector(img_path):
  index = vgg_model(img_path)
  if index>=151 and index<=268:
    return True
  else: False

# Set PIL to be tolerant of image files that are truncated.
ImageFile.LOAD_TRUNCATED_IMAGES = True

train_dir = 'dogImages/train'
test_dir = 'dogImages/test'
valid_dir = 'dogImages/valid'

train_folder = datasets.ImageFolder(train_dir)

model = models.resnet152(pretrained=True)

# Freeze training for all "feature" layers -> turning off computing gradient for each parameter
for param in model.parameters():
  param.requires_grad_(False)

# Building classifier
# Next adding my own layers after features layers for 
# training purpose which is to be customised according to output labels available
pre_trained_classifier = nn.Sequential(
    OrderedDict(
        [
         ('fc1', nn.Linear(2048, 1000)),
         ('relu', nn.ReLU()),
         ('dropout', nn.Dropout(p=0.5)),
         ('fc2', nn.Linear(1000, 133)),
         ('output', nn.LogSoftmax(dim=1))        
        ]
    )
)

model.fc = pre_trained_classifier

# move model to GPU if CUDA is available
if use_cuda:
    model = model.cuda()

loss_fun = nn.CrossEntropyLoss()
optimizer = optim.SGD(model.fc.parameters(), lr=0.01, momentum=0.9)
scheduler = optim.lr_scheduler.ExponentialLR(optimizer, gamma=0.9)


# load the model that got the best validation accuracy
model.load_state_dict(torch.load('save_trained_model.pt', map_location=torch.device('cpu')))

# list of class names (breed names of dogs)
class_names_breed = [breed[4:].replace("_", " ") for breed in train_folder.classes]

# function that takes a dog image & returns the breed of that dog in the image
def predict_dog_breed(image_path):
  # img = Image.open(image_path)
  img = image_path
  
  model.to('cpu')

  # pre-processing the input image
  transform_image = transforms.Compose(
      [ transforms.Resize(256),
       transforms.CenterCrop(224),
       transforms.ToTensor()    
      ]
  )

  processed_image = transform_image(img).unsqueeze(0)

  # feedforward : feeding to trained model
  output = model(processed_image)

  # taking the prediction 
  _, pred = torch.max(output, dim=1)

  return class_names_breed[pred[0]]

# checking if prediction is correct or not
def run_app(image_path):
  # img = Image.open(image_path)
  img = image_path
  out_str = ""

  ### handle cases for a human face, dog, and neither
  if dog_detector(image_path) == True:
    out_str = "Hi, This Dog's Breed is " + str(predict_dog_breed(image_path))
    return out_str
  elif detector_humanface(image_path) == True:
    out_str = "hello, human! You look like a " + str(predict_dog_breed(image_path)) + " Breed"
    return out_str
  else:
    out_str = 'Error... No Dog or Human Face present!! Nothing Detected!!'
    return out_str
    
def set_example_image(example: list) -> dict:
  return gr.Image.update(value=example[0])

demo = gr.Blocks()

with demo:
    gr.Markdown(
    """
    # <center>Breed Finder !</center>
    
    ### Find the breed for dog image or resembling breed for human Image!

    Enter the image of a dog or human and check its resembling breed...
    1. If uploaded image is of Dog : it will give its Breed
    2. Else If uploaded image is of Human: it will give its resembling breed of dog
    """)
    
    inp = gr.Image(label='input image of dog/human', type='pil')
    
    with gr.Row():
         paths = sorted(pathlib.Path('images').glob('*.jpg'))
         example_images = gr.Dataset(components=[inp],
                                     samples=[[path.as_posix()]
                                              for path in paths])
  
    submit = gr.Button("Generate Breed Name")

    submit.click(fn=run_app,
                 inputs=inp,
                 outputs=gr.Textbox(label='breed'))
                 
    example_images.click(fn=set_example_image,
                             inputs=example_images,
                             outputs=example_images.components)
                             
    gr.Markdown(
            '<center><img src="https://visitor-badge.glitch.me/badge?page_id=gradio-blocks.dog-breed-identification-app" alt="visitor badge"/></center>'
        )

demo.launch(enable_queue=True, debug=True)