import pandas as pd
import numpy as np
import sklearn
from sklearn.metrics.pairwise import cosine_similarity

# Load movie metadata into a pandas DataFrame
movie_data = pd.read_csv('movies_metadata.csv', usecols=['title', 'budget', 'revenue', 'vote_count', 'vote_average'])

# Load movie ratings into a pandas DataFrame
ratings_data = pd.read_csv('ratings_small.csv')

# Collaborative Filtering
def collaborative_filtering(user_id, num_recommendations=5):
    user_ratings = ratings_data[ratings_data['userId'] == user_id]
    user_ratings = user_ratings.merge(movie_data[['title']], left_on='movieId', right_index=True)
    user_ratings = user_ratings.drop_duplicates(subset=['title'], keep='first')

    # Get the mean ratings for each movie
    movie_ratings_mean = ratings_data.groupby('movieId')['rating'].mean().reset_index()

    # Merge with movie data to get movie titles
    movie_ratings_mean = movie_ratings_mean.merge(movie_data[['title']], left_on='movieId', right_index=True)
    movie_ratings_mean = movie_ratings_mean.drop_duplicates(subset=['title'], keep='first')

    # Sort movies by mean rating in descending order
    top_movies = movie_ratings_mean.sort_values(by='rating', ascending=False)['title'][:num_recommendations]

    return top_movies.tolist()

# Content-Based Filtering
def content_based_filtering(movie_title, num_recommendations=5):
    # Drop duplicates and missing values based on 'title' column
    movie_data_unique = movie_data.drop_duplicates(subset='title', keep='first').dropna(subset=['budget', 'revenue', 'vote_count', 'vote_average'])

    # Calculate similarity scores between input movie and other movies based on selected features
    selected_features = ['budget', 'revenue', 'vote_count', 'vote_average']
    movie_features = movie_data_unique[selected_features]
    similarity_scores = cosine_similarity(movie_features[movie_data_unique['title'] == movie_title], movie_features)
    movie_data_unique['similarity'] = similarity_scores.flatten()

    # Get the most similar movies
    top_movies = movie_data_unique[movie_data_unique['title'] != movie_title].sort_values(by='similarity', ascending=False)['title'][:num_recommendations]

    return top_movies.tolist()


# Hybrid Method
def hybrid_filtering(user_id, movie_title, num_recommendations=5):
    collaborative_movies = collaborative_filtering(user_id)
    content_based_movies = content_based_filtering(movie_title)

    # Merge and deduplicate the recommendations
    all_recommendations = collaborative_movies + content_based_movies
    unique_recommendations = list(set(all_recommendations))[:num_recommendations]

    return unique_recommendations
import gradio as gr

with gr.Blocks() as demo:
    gr.Markdown("Flip text or image files using this demo.")
    title="Movie Recommender System"
    user_id = gr.Textbox(placeholder='user_id')
    movie_title = gr.Textbox(placeholder='movie_title')

    #input2 = gr.Textbox(placeholder='second film name...')

    #input3 = gr.Textbox(placeholder='third film name...')

    text_output=gr.Textbox( )


    text_button = gr.Button("Submit")
    text_button.click(hybrid_filtering, inputs= [ user_id,movie_title], outputs=text_output)
demo.launch()