import os
import numpy as np
import tensorflow as tf
from tensorflow.keras.layers import Input, Dense, Conv2D, Conv2DTranspose, Flatten, Reshape, MaxPooling2D
from tensorflow.keras.models import Model, load_model
from tensorflow.keras.preprocessing.image import load_img, img_to_array
from PIL import Image
import gradio as gr

# Путь к папке с изображениями
IMAGE_DIR = "dataset"
# Размер изображения для обучения
IMAGE_SIZE = (144, 144)

# Загрузка и предобработка изображений
def load_images(image_dir):
    images = []
    for filename in os.listdir(image_dir):
        try:
            image = load_img(os.path.join(image_dir, filename), target_size=IMAGE_SIZE)
            image = img_to_array(image) / 255.0
            images.append(image)
        except:
            pass
    return np.array(images)

# Создание и обучение SQVAE модели
def train_sqvae(images):
    input_shape = images[0].shape
    latent_dim = 512

    # Encoder
    input_img = Input(shape=input_shape)
    x = Conv2D(32, 3, padding='same', activation='relu')(input_img)
    x = Conv2D(64, 3, padding='same', activation='relu')(x)
    x = MaxPooling2D(pool_size=(2, 2))(x)

    x = Conv2D(128, 3, padding='same', activation='relu')(x)
    x = Conv2D(256, 3, padding='same', activation='relu')(x)
    x = MaxPooling2D(pool_size=(2, 2))(x)

    x = Conv2D(512, 3, padding='same', activation='relu')(x)
    x = Conv2D(1024, 3, padding='same', activation='relu')(x)
    x = MaxPooling2D(pool_size=(2, 2))(x)

    # Latent space
    x = Flatten()(x)
    latent = Dense(latent_dim, activation='relu')(x)
    encoder = Model(input_img, latent, name='encoder')

    # Decoder
    latent_input = Input(shape=(latent_dim,))
    x = Dense(128 * int(input_shape[0] / 16) * int(input_shape[1] / 16), activation='relu')(latent_input)
    x = Reshape((int(input_shape[0] / 16), int(input_shape[1] / 16), 128))(x)

    x = Conv2DTranspose(256, 3, strides=(2, 2), padding='same', activation='relu')(x)
    x = Conv2D(256, 3, padding='same', activation='relu')(x)
    x = Conv2D(256, 3, padding='same', activation='relu')(x)

    x = Conv2DTranspose(128, 3, strides=(2, 2), padding='same', activation='relu')(x)
    x = Conv2D(128, 3, padding='same', activation='relu')(x)
    x = Conv2D(128, 3, padding='same', activation='relu')(x)

    x = Conv2DTranspose(64, 3, strides=(2, 2), padding='same', activation='relu')(x)
    x = Conv2D(64, 3, padding='same', activation='relu')(x)
    x = Conv2D(64, 3, padding='same', activation='relu')(x)

    x = Conv2DTranspose(32, 3, strides=(2, 2), padding='same', activation='relu')(x)
    x = Conv2D(32, 3, padding='same', activation='relu')(x)
    x = Conv2D(32, 3, padding='same', activation='relu')(x)

    x = Conv2DTranspose(16, 3, strides=(2, 2), padding='same', activation='relu')(x)
    x = Conv2D(16, 3, padding='same', activation='relu')(x)
    x = Conv2D(16, 3, padding='same', activation='relu')(x)

    x = Conv2DTranspose(8, 3, strides=(2, 2), padding='same', activation='relu')(x)
    x = Conv2D(8, 3, padding='same', activation='relu')(x)
    x = Conv2D(8, 3, padding='same', activation='relu')(x)

    decoded = Conv2D(3, 3, padding='same', activation='sigmoid')(x)
    decoder = Model(latent_input, decoded, name='decoder')

    autoencoder = Model(input_img, decoder(encoder(input_img)), name='autoencoder')
    from tensorflow.keras.metrics import MeanSquaredError, MeanAbsolutePercentageError

    autoencoder.compile(optimizer='adam', loss='binary_crossentropy', metrics=[MeanSquaredError(), MeanAbsolutePercentageError()])
    autoencoder.summary()
    autoencoder.fit(images, images, epochs=50, batch_size=64)

    encoder.save("encoder_model.h5")
    decoder.save("decoder_model.h5")

    return encoder, decoder

# Загрузка и предобработка данных для обучения
#images = load_images(IMAGE_DIR)

# Обучение модели и сохранение encoder и decoder
#encoder, decoder = train_sqvae(images)

import cv2

def generate_image(seed, mode="random", steps=1):
    random_noise = np.random.randint(0, 256, size=(1, 4096))
    if mode == "random":
        decoder = load_model("decoder_model.h5")
        generated_image = decoder.predict(random_noise)[0]
    elif mode == "interpolation":
        decoder = load_model("decoder_model.h5")
        latent_a = np.random.normal(size=(1, 256))
        latent_b = np.random.normal(size=(1, 256))
        alpha = np.linspace(0, 1, num=10)
        latents = alpha[:, np.newaxis] * latent_a + (1 - alpha[:, np.newaxis]) * latent_b
        generated_image = decoder.predict(latents)
        generated_image = np.concatenate(generated_image, axis=1)
    elif mode == "reconstruction":
        encoder = load_model("encoder_model.h5")
        decoder = load_model("decoder_model.h5")
        image = images[seed]
        latent = encoder.predict(image[np.newaxis, ...])
        generated_image = decoder.predict(latent)[0]
        generated_image = (generated_image * 255).astype(np.uint8)

    for _ in range(int(steps)-1):
        generated_image = decoder.predict(generated_image[np.newaxis, ...])[0]
        generated_image = (generated_image * 255).astype(np.uint8)

        return generated_image

inputs = [
    gr.inputs.Slider(minimum=0, maximum=100, default=0, label="Seed"),
    gr.inputs.Radio(["random", "interpolation", "reconstruction"], label="Mode"),
    gr.inputs.Number(default=1, label="Steps")
]
outputs = gr.outputs.Image(type='numpy')

gr.Interface(fn=generate_image, inputs=inputs, outputs=outputs).launch(debug=True)