from contextlib import nullcontext
import torch
from torch import autocast
from diffusers import StableDiffusionPipeline
import gradio as gr


CHECKPOINTS = [
    "epoch-000025",
    "epoch-000081"
]

device = "cuda" if torch.cuda.is_available() else "cpu"
context = autocast if device == "cuda" else nullcontext
dtype = torch.float16 if device == "cuda" else torch.float32

def load_pipe(checkpoint):

    pipe = StableDiffusionPipeline.from_pretrained("Gazoche/sd-gundam-diffusers", revision=checkpoint, torch_dtype=dtype)
    pipe = pipe.to(device)

    # Disabling the NSFW filter as it's getting confused by the generated images
    def null_safety(images, **kwargs):
        return images, False
    pipe.safety_checker = null_safety

    return pipe

pipes = {
    checkpoint: load_pipe(checkpoint)
    for checkpoint in CHECKPOINTS
}

def infer(prompt, n_samples, steps, scale, model):

    checkpoint = "epoch-000025" if model == "normal" else "epoch-000081"

    in_prompt = ""
    guidance_scale = 0.0
    if prompt is not None:
        in_prompt = prompt
        guidance_scale = scale

    with context("cuda"):
        images = pipes[checkpoint](
            n_samples * [in_prompt],
            guidance_scale=guidance_scale,
            num_inference_steps=steps
        ).images

    return images


def infer_random(n_samples, steps, scale, model):
    return infer(None, n_samples, steps, scale, model)

css = """
        a {
            color: inherit;
            text-decoration: underline;
        }
        .gradio-container {
            font-family: 'IBM Plex Sans', sans-serif;
        }
        .gr-button {
            color: white;
            border-color: #9d66e5;
            background: #9d66e5;
        }
        input[type='range'] {
            accent-color: #9d66e5;
        }
        .dark input[type='range'] {
            accent-color: #dfdfdf;
        }
        .container {
            max-width: 730px;
            margin: auto;
            padding-top: 1.5rem;
        }
        #gallery {
            min-height: 22rem;
            margin-bottom: 15px;
            margin-left: auto;
            margin-right: auto;
            border-bottom-right-radius: .5rem !important;
            border-bottom-left-radius: .5rem !important;
        }
        #gallery>div>.h-full {
            min-height: 20rem;
        }
        .details:hover {
            text-decoration: underline;
        }
        .gr-button {
            white-space: nowrap;
        }
        .gr-button:focus {
            border-color: rgb(147 197 253 / var(--tw-border-opacity));
            outline: none;
            box-shadow: var(--tw-ring-offset-shadow), var(--tw-ring-shadow), var(--tw-shadow, 0 0 #0000);
            --tw-border-opacity: 1;
            --tw-ring-offset-shadow: var(--tw-ring-inset) 0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color);
            --tw-ring-shadow: var(--tw-ring-inset) 0 0 0 calc(3px var(--tw-ring-offset-width)) var(--tw-ring-color);
            --tw-ring-color: rgb(191 219 254 / var(--tw-ring-opacity));
            --tw-ring-opacity: .5;
        }
        #advanced-options {
            margin-bottom: 20px;
        }
        .footer {
            margin-bottom: 45px;
            margin-top: 35px;
            text-align: center;
            border-bottom: 1px solid #e5e5e5;
        }
        .footer>p {
            font-size: .8rem;
            display: inline-block;
            padding: 0 10px;
            transform: translateY(10px);
            background: white;
        }
        .dark .logo{ filter: invert(1); }
        .dark .footer {
            border-color: #303030;
        }
        .dark .footer>p {
            background: #0b0f19;
        }
        .acknowledgments h4{
            margin: 1.25em 0 .25em 0;
            font-weight: bold;
            font-size: 115%;
        }
"""

block = gr.Blocks(css=css)

with block:
    gr.HTML(
        """
            <div style="text-align: center; max-width: 650px; margin: 0 auto;">
              <div>
                <h1 style="font-weight: 900; font-size: 3rem;">
                  Gundam text to image
                </h1>
              </div>
              <p style="margin-bottom: 10px; font-size: 94%">
              From a text description, generate a mecha from the anime franchise Mobile Suit Gundam
              </p>
              <p style="margin-bottom: 10px; font-size: 94%">
              Github: <a href="https://github.com/Askannz/gundam-stable-diffusion">https://github.com/Askannz/gundam-stable-diffusion</a>
              </p>
               <ul>
                <li>More steps generally means less visual noise but fewer details</li>
                <li>Text guidance controls how much the prompt influences the generation</li>
                <li>The overfitted model gives cleaner but less original results</li>
              </ul>
            </div>
        """
    )
    with gr.Group():

        with gr.Box():
            with gr.Row().style(mobile_collapse=False, equal_height=True):
                text = gr.Textbox(
                    label="Enter your prompt",
                    show_label=False,
                    max_lines=1,
                    placeholder="Enter your prompt",
                ).style(
                    border=(True, False, True, True),
                    rounded=(True, False, False, True),
                    container=False,
                )
                btn = gr.Button("Generate from prompt").style(
                    margin=False,
                    rounded=(False, True, True, False),
                )

        with gr.Box():
            with gr.Row().style(mobile_collapse=False, equal_height=True):
                btn_rand = gr.Button("Random").style(
                    margin=False,
                    rounded=(False, True, True, False),
                )

        gallery = gr.Gallery(
            label="Generated images", show_label=False, elem_id="gallery"
        ).style(grid=[2], height="auto")


        with gr.Row(elem_id="advanced-options"):
            samples = gr.Slider(label="Images", minimum=1, maximum=2, value=1, step=1)
            steps = gr.Slider(label="Steps", minimum=5, maximum=50, value=25, step=5)
            scale = gr.Slider(
                label="Text Guidance Scale", minimum=0, maximum=50, value=7.5, step=0.1
            )

        with gr.Row(elem_id="checkpoint"):
            model = gr.Radio(label="Model", choices=["normal", "overfitted"], value="normal")
            #model = gr.Radio(label="Model", choices=["normal"], value="normal")


        text.submit(infer, inputs=[text, samples, steps, scale, model], outputs=gallery)
        btn.click(infer, inputs=[text, samples, steps, scale, model], outputs=gallery)
        btn_rand.click(infer_random, inputs=[samples, steps, scale, model], outputs=gallery)
        gr.HTML(
            """
                <div class="footer">
                    <p> Gradio Demo by 🤗 Hugging Face and Gazoche
                    </p>
                </div>
           """
        )

block.launch()
