#!/usr/bin/env python

from __future__ import annotations
import argparse
import functools
import os
import pathlib
import sys
from typing import Callable
import uuid

import gradio as gr
import huggingface_hub
import numpy as np
import PIL.Image

from io import BytesIO
from wbc.cartoonize import Cartoonize

ORIGINAL_REPO_URL = 'https://github.com/SystemErrorWang/White-box-Cartoonization'
TITLE = 'SystemErrorWang/White-box-Cartoonization'
DESCRIPTION = f"""This is a demo for {ORIGINAL_REPO_URL}.

"""
ARTICLE = """

"""

SAFEHASH = [x for x in "0123456789-abcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXYZ"]
def compress_UUID():
    '''
    根据http://www.ietf.org/rfc/rfc1738.txt,由uuid编码扩bai大字符域生成du串
    包括:[0-9a-zA-Z\-_]共64个
    长度:(32-2)/3*2=20
    备注:可在地球上人zhi人都用,使用100年不重复(2^120)
    :return:String
    '''
    row = str(uuid.uuid4()).replace('-', '')
    safe_code = ''
    for i in range(10):
        enbin = "%012d" % int(bin(int(row[i * 3] + row[i * 3 + 1] + row[i * 3 + 2], 16))[2:], 10)
        safe_code += (SAFEHASH[int(enbin[0:6], 2)] + SAFEHASH[int(enbin[6:12], 2)])
    safe_code = safe_code.replace('-', '')
    return safe_code


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser()
    parser.add_argument('--device', type=str, default='cpu')
    parser.add_argument('--theme', type=str)
    parser.add_argument('--live', action='store_true')
    parser.add_argument('--share', action='store_true')
    parser.add_argument('--port', type=int)
    parser.add_argument('--disable-queue',
                        dest='enable_queue',
                        action='store_false')
    parser.add_argument('--allow-flagging', type=str, default='never')
    parser.add_argument('--allow-screenshot', action='store_true')
    return parser.parse_args()

def run(
        image,
        cartoonize : Cartoonize
) -> tuple[PIL.Image.Image]:

    out_path = compress_UUID()+'.png'
    cartoonize.run_sigle(image.name, out_path)

    return PIL.Image.open(out_path)


def main():
    gr.close_all()

    args = parse_args()

    cartoonize = Cartoonize(os.path.join(os.path.dirname(os.path.abspath(__file__)),'wbc/saved_models/'))

    func = functools.partial(run, cartoonize=cartoonize)
    func = functools.update_wrapper(func, run)

    gr.Interface(
        func,
        [
            gr.inputs.Image(type='file', label='Input Image'),
        ],
        [
            gr.outputs.Image(
                type='pil',
                label='Result'),
        ],
        # examples=examples,
        theme=args.theme,
        title=TITLE,
        description=DESCRIPTION,
        article=ARTICLE,
        allow_screenshot=args.allow_screenshot,
        allow_flagging=args.allow_flagging,
        live=args.live,
    ).launch(
        enable_queue=args.enable_queue,
        server_port=args.port,
        share=args.share,
    )


if __name__ == '__main__':
    main()
