import logging
import pathlib
import gradio as gr
import pandas as pd
from gt4sd.algorithms.generation.polymer_blocks import (
    PolymerBlocksGenerator,
    PolymerBlocks,
)

from gt4sd.algorithms.registry import ApplicationsRegistry

from utils import draw_grid_generate

logger = logging.getLogger(__name__)
logger.addHandler(logging.NullHandler())


def run_inference(algorithm_version: str, length: float, number_of_samples: int):

    config = PolymerBlocksGenerator(
        algorithm_version=algorithm_version,
        batch_size=32,
        generated_length=length,
    )
    model = PolymerBlocks(config)
    samples = list(model.sample(number_of_samples))

    return draw_grid_generate(samples=samples, n_cols=5, seeds=[])


if __name__ == "__main__":

    # Preparation (retrieve all available algorithms)
    all_algos = ApplicationsRegistry.list_available()
    algos = [
        x["algorithm_version"]
        for x in list(
            filter(lambda x: "PolymerBlocks" in x["algorithm_name"], all_algos)
        )
    ]

    # Load metadata
    metadata_root = pathlib.Path(__file__).parent.joinpath("model_cards")

    examples = pd.read_csv(metadata_root.joinpath("examples.csv"), header=None).fillna(
        ""
    )

    with open(metadata_root.joinpath("article.md"), "r") as f:
        article = f.read()
    with open(metadata_root.joinpath("description.md"), "r") as f:
        description = f.read()

    demo = gr.Interface(
        fn=run_inference,
        title="Polymer Blocks",
        inputs=[
            gr.Dropdown(algos, label="Algorithm version", value="v0"),
            gr.Slider(
                minimum=5,
                maximum=400,
                value=100,
                label="Maximal sequence length",
                step=1,
            ),
            gr.Slider(
                minimum=1, maximum=50, value=10, label="Number of samples", step=1
            ),
        ],
        outputs=gr.HTML(label="Output"),
        article=article,
        description=description,
        examples=examples.values.tolist(),
    )
    demo.launch(debug=True, show_error=True)
