import os
import gradio as gr
import pandas as pd
import torch
import torch.nn as nn
import transformers
from transformers import AutoTokenizer, AutoConfig, LlamaForCausalLM, LlamaTokenizer, GenerationConfig, AutoModel, pipeline
import pandas as pd
import tensorflow as tf
import numpy as np
import math
import time
import csv
import nltk
from nltk.tokenize import word_tokenize
from nltk.corpus import stopwords
nltk.download('stopwords')
nltk.download('punkt')
import string
import huggingface_hub
from huggingface_hub import Repository
from datetime import datetime

########### Import Classifier Embeddings #########
class_embeddings = pd.read_csv('Embeddings/MainClassEmbeddings.csv')

########### DATA CLEANER VARIABLES #############
all_stopwords = stopwords.words('english') # Making sure to only use English stopwords
extra_stopwords = ['ii', 'iii'] # Can add extra stopwords to be removed from dataset/input abstracts
all_stopwords.extend(extra_stopwords)

modelpath = os.environ.get("MODEL_PATH")

########### GET CLAIMED TRAINED MODEL ###########
tokenizer = LlamaTokenizer.from_pretrained(modelpath)

model = LlamaForCausalLM.from_pretrained(
    modelpath,
    load_in_8bit=True,
    device_map='auto',
)

HF_TOKEN = os.environ.get("HF_TOKEN")

DATASET_REPO_URL = "https://huggingface.co/datasets/thepolymerguy/logger"
DATA_FILENAME = "data.csv"
DATA_FILE = os.path.join("data", DATA_FILENAME)

repo = Repository(
    local_dir="data", clone_from=DATASET_REPO_URL, use_auth_token=HF_TOKEN
)

def store_log():
    with open(DATA_FILE, "a") as csvfile:
        writer = csv.DictWriter(csvfile, fieldnames=["count", "time"])
        writer.writerow(
            {"count": 1, "time": str(datetime.now())}
        )
    commit_url = repo.push_to_hub()
    print(commit_url)
    return 

########## DEFINING FUNCTIONS ###################

def mean_pooling(model_output, attention_mask):
    token_embeddings = model_output[0] 
    input_mask_expanded = attention_mask.unsqueeze(-1).expand(token_embeddings.size()).float()
    return tf.reduce_sum(token_embeddings * input_mask_expanded, 1) / tf.clip_by_value(input_mask_expanded.sum(1), clip_value_min=1e-9, clip_value_max=math.inf)
    
def broad_scope_class_predictor(class_embeddings, abstract_embedding, SearchType, N=5, Sensitivity='Medium'):
    predictions = pd.DataFrame(columns=['Class Name', 'Score'])
    for i in range(len(class_embeddings)):
        class_name = class_embeddings.iloc[i, 0]
        embedding = class_embeddings.iloc[i, 2]
        embedding = convert_saved_embeddings(embedding)
        abstract_embedding = abstract_embedding.numpy()
        abstract_embedding = torch.from_numpy(abstract_embedding)
        cos = torch.nn.CosineSimilarity(dim=1)
        score = cos(abstract_embedding, embedding).numpy().tolist()
        result = [class_name, score[0]]
        predictions.loc[len(predictions)] = result
    if Sensitivity == 'High':
        Threshold = 0.5
    elif Sensitivity == 'Medium':
        Threshold = 0.40
    elif Sensitivity == 'Low':
        Threshold = 0.35
    GreenLikelihood = 'False'
    HighestSimilarity = predictions.nlargest(N, ['Score'])
    HighestSimilarity = HighestSimilarity['Class Name'].tolist()
    HighestSimilarityClass = [x.split('/')[0] for x in HighestSimilarity]
    if SearchType == 'Google Patent Search':
      Links = [f'https://patents.google.com/?q=({x}%2f00)&oq={x}%2f00' for x in HighestSimilarityClass]
    elif SearchType == 'Espacenet Patent Search':
      Links = [f'https://worldwide.espacenet.com/patent/search?q=cpc%3D{x}%2F00%2Flow' for x in HighestSimilarityClass]    
    HighestSimilarity = pd.DataFrame({'Class':HighestSimilarity, 'Links':Links})
    return HighestSimilarity


def sentence_embedder(sentences, model_path):
  tokenizer = AutoTokenizer.from_pretrained(model_path) #instantiating the sentence embedder using HuggingFace library
  model = AutoModel.from_pretrained(model_path, from_tf=True) #making a model instance
  encoded_input = tokenizer(sentences, padding=True, truncation=True, return_tensors='pt')
  # Compute token embeddings
  with torch.no_grad():
    model_output = model(**encoded_input)
  sentence_embeddings = mean_pooling(model_output, encoded_input['attention_mask']) #outputs a (1, 384) tensor representation of input text
  return sentence_embeddings
    

def add_text(history, text):
    history = history + [(text, None)]
    return history, ""


def convert_saved_embeddings(embedding_string):
    """
    Preparing pre-computed embeddings for use for comparison with new abstract embeddings .
    Pre-computed embeddings are saved as tensors in string format so need to be converted back to numpy arrays in order to calculate cosine similarity.
    :param embedding_string:
    :return: Should be a single tensor with dims (,384) in string format
    """
    embedding = embedding_string.replace('(', '')
    embedding = embedding.replace(')', '')
    embedding = embedding.replace('[', '')
    embedding = embedding.replace(']', '')
    embedding = embedding.replace('tensor', '')
    embedding = embedding.replace(' ', '')
    embedding = embedding.split(',')
    embedding = [float(x) for x in embedding]
    embedding = np.array(embedding)
    embedding = np.expand_dims(embedding, axis=0)
    embedding = torch.from_numpy(embedding)
    return embedding

########## LOADING PRE-COMPUTED EMBEDDINGS ##########

def clean_data(input, type='Dataframe'):
    if type == 'Dataframe':
        cleaneddf = pd.DataFrame(columns=['Class', 'Description'])
        for i in range(0, len(input)):
            row_list = input.loc[i, :].values.flatten().tolist()
            noNaN_row = [x for x in row_list if str(x) != 'nan']
            listrow = []
            if len(noNaN_row) > 0:
                row = noNaN_row[:-1]
                row = [x.strip() for x in row]
                row = (" ").join(row)
                text_tokens = word_tokenize(row)  # splits abstracts into individual tokens to allow removal of stopwords by list comprehension
                Stopword_Filtered_List = [word for word in text_tokens if not word in all_stopwords]  # removes stopwords
                row = (" ").join(Stopword_Filtered_List)  # returns abstract to string form
                removechars = ['[', ']', '{', '}', ';', '(', ')', ',', '.', ':', '/', '-', '#', '?', '@', '£', '$']
                for char in removechars:
                    row = list(map(lambda x: x.replace(char, ''), row))

                row = ''.join(row)
                wnum = row.split(' ')
                wnum = [x.lower() for x in wnum]
                #remove duplicate words
                wnum = list(dict.fromkeys(wnum))
                #removing numbers
                wonum = []
                for x in wnum:
                    xv = list(x)
                    xv = [i.isnumeric() for i in xv]
                    if True in xv:
                        continue
                    else:
                        wonum.append(x)
                row = ' '.join(wonum)
                l = [noNaN_row[-1], row]
                cleaneddf.loc[len(cleaneddf)] = l
        cleaneddf = cleaneddf.drop_duplicates(subset=['Description'])
        cleaneddf.to_csv('E:/Users/eeo21/Startup/CPC_Classifications_List/additionalcleanedclasses.csv', index=False)
        return cleaneddf

    elif type == 'String':
        text_tokens = word_tokenize(input)  # splits abstracts into individual tokens to allow removal of stopwords by list comprehension
        Stopword_Filtered_List = [word for word in text_tokens if not word in all_stopwords]  # removes stopwords
        row = (" ").join(Stopword_Filtered_List)  # returns abstract to string form
        removechars = ['[', ']', '{', '}', ';', '(', ')', ',', '.', ':', '/', '-', '#', '?', '@', '£', '$']
        for char in removechars:
            row = list(map(lambda x: x.replace(char, ''), row))
        row = ''.join(row)
        wnum = row.split(' ')
        wnum = [x.lower() for x in wnum]
        # remove duplicate words
        wnum = list(dict.fromkeys(wnum))
        # removing numbers
        wonum = []
        for x in wnum:
            xv = list(x)
            xv = [i.isnumeric() for i in xv]
            if True in xv:
                continue
            else:
                wonum.append(x)
        row = ' '.join(wonum)
        return row

def classifier(userin, SearchType):
    clean_in = clean_data(userin, type='String')
    in_emb = sentence_embedder(clean_in, 'Model_bert')
    
    Number = 10
    broad_scope_predictions = broad_scope_class_predictor(class_embeddings, in_emb, SearchType, Number, Sensitivity='High')

    class_links = []
    for i in range(Number):
      class_links.append("[[{}]]({})".format(broad_scope_predictions['Class'][i], broad_scope_predictions['Links'][i]))


    md_class = '\n'.join(class_links)

    store_log()

    return md_class

def generateresponse(history, temp, top_p, tokens):
    
    global model
    global tokenizer

    user = history[-1][0]

    PROMPT = f"""Below is an instruction that describes a task. Write a response that appropriately completes the request.
    ### Instruction:
    {user}
    ### Response:"""

    pipe = pipeline(
        "text-generation",
        model=model, 
        tokenizer=tokenizer, 
        max_length=tokens,
        temperature=temp,
        top_p=top_p,
        repetition_penalty=1.15
    )

    outputs = pipe(PROMPT)
    outputs = outputs[0]['generated_text']
    outputs = str(outputs).split('### Response')[1]
    
    response = f"Response{outputs}"

    store_log()
    
    return response

def run_model(userin, dropd, temp, top_p, tokens):

    global model
    global tokenizer

    if dropd in ["An apparatus", "A method of use", "A method", "A method of manufacturing", "A system"]:
        PROMPT = claim_selector(userin, dropd)
    elif dropd in ["Generate a Detailed Description Paragraph", "Generate a Abstract", "What are the Benefits/Technical Effects"]:
        PROMPT = desc_selector(userin, dropd)

    pipe = pipeline(
        "text-generation",
        model=model, 
        tokenizer=tokenizer, 
        max_length=tokens,
        temperature=temp,
        top_p=top_p,
        repetition_penalty=1.15
    )

    outputs = pipe(PROMPT)

    outputs = outputs[0]['generated_text']
    outputs = str(outputs).split('### Response')[1]
    outputs = outputs.split('\n    \n    \n    \n*')[0]
    
    response = f"Response{outputs}"

    store_log()
    
    return response

def prosecute(application, priorart, temp, top_p, tokens):

    global model
    global tokenizer

    pipe = pipeline(
        "text-generation",
        model=model, 
        tokenizer=tokenizer, 
        max_length=tokens,
        temperature=temp,
        top_p=top_p,
        repetition_penalty=1.15
    )

    PROMPT = f"""
    Draft an argument for the patentability in favour of the application using the European Patent Office Problem Solution appraoch by summarising the difference between the Application and the prior art. If there is no differnce, say that the present invention is not novel/inventive.

    Application: {application}

    Prior Art: {priorart}

    ### Response: The objective technical problem solved by the present invention"""

    outputs = pipe(PROMPT)

    outputs = outputs[0]['generated_text']
    outputs = str(outputs).split('### Response')[1]
    outputs = outputs.split('\n    \n    \n    \n*')[0]
    
    response = f"Response{outputs}"

    store_log()
    
    return response

def ideator(userin, temp, top_p, tokens):

    global model
    global tokenizer

    pipe = pipeline(
        "text-generation",
        model=model, 
        tokenizer=tokenizer, 
        max_length=tokens,
        temperature=temp,
        top_p=top_p,
        repetition_penalty=1.15
    )

    PROMPT = f"""
    How can I make {userin}

    ### Response: You could implement the invention as follows:"""

    outputs = pipe(PROMPT)

    outputs = outputs[0]['generated_text']
    outputs = str(outputs).split('### Response')[1]
    outputs = outputs.split('\n    \n    \n    \n*')[0]

    
    response = f"Response{outputs}"

    store_log()
    
    return response

def Chat(userin, temp, top_p, tokens):

    global model
    global tokenizer

    pipe = pipeline(
        "text-generation",
        model=model, 
        tokenizer=tokenizer, 
        max_length=tokens,
        temperature=temp,
        top_p=top_p,
        repetition_penalty=1.15
    )

    PROMPT = f"""Below is a query from a user. Respond appropriately to the query.
    ### Query:
    {userin}
    ### Response:"""

    outputs = pipe(PROMPT)

    outputs = outputs[0]['generated_text']
    outputs = str(outputs).split('### Response')[1]
    outputs = outputs.split('\n    \n    \n    \n*')[0]
    
    response = f"Response{outputs}"

    store_log()
    
    return response
       
def claim_selector(userin, dropd):
    
    PROMPT = f"""
    Draft a patent claim 1 for {dropd} for the following invention: {userin}
    ### Response:{dropd} comprising:"""

    return PROMPT

def desc_selector(userin, dropd):
    
    PROMPT = f"""
    {dropd} for a patent application for the following invention: {userin}
    ### Response:"""

    return PROMPT

############# GRADIO APP ###############

theme = gr.themes.Base(
    primary_hue="indigo",
).set(
    prose_text_size='*text_sm'
)

with gr.Blocks(title='Patent Toolkit', theme=theme) as demo:
    
    gr.Markdown("""
    # GENERATIVE TOOLKIT FOR PATENT ATTORNEYS AND INVENTORS
    The patenting process can be complex, time-consuming and expensive. We believe that AI will one day alleviate these problems.
    
    As a proof of concept, we've trained Meta's Llama on over 200k entries, with a focus on tasks related to the intellectual property domain.

    We are currently running this demo on a less powerful version of our model due to computational limitations. If you would like to see our most powerful model in action, please contact us at this email: e.ogbomo21@imperial.ac.uk
    
    We know that confidentiality is probably the number one concern for attorneys when considering using such tools. We don't store any of your inputs to use for further training and we don't use the OpenAI API (ChatGPT) as our backend, meaning that confidentiality is not comprimised!  
    
    Please note that this is for research purposes and shouldn't be used commercially. 
    
    None of the outputs of this model, taken in part or in its entirety, constitutes legal advice. If you are seeking protection for you intellectual property, consult a registered patent/trademark attorney.
    """)

    # with gr.Tab("Ideator"):
    #     gr.Markdown(""" 
    #     Use this tool to generate ideas for how to implement an invention/creation.
    #      """)
    #     with gr.Row(scale=1, min_width=600):
    #         with gr.Column():
    #           userin = gr.Text(label="Input", lines=5)
    #         with gr.Column():
    #           text2 = gr.Textbox(label="Output", lines=5)
    #     with gr.Row():
    #         btn = gr.Button("Submit")
    #     with gr.Row():
    #       with gr.Accordion("Parameters"):
    #           temp = gr.Slider(minimum=0, maximum=1, value=0.6, label="Temperature", step=0.1)
    #           top_p = gr.Slider(minimum=0.5, maximum=1, value=0.95, label="Top P", step=0.1)
    #           tokens = gr.Slider(minimum=5, maximum=2058, value=512, label="Max Tokens", step=1)

    #     btn.click(fn=ideator, inputs=[userin, temp, top_p, tokens], outputs=text2)

    with gr.Tab("Claim Drafter"):
        gr.Markdown(""" 
        Use this tool to expand your idea into the technical language of a patent claim. You can specify the type of claim you want using the dropdown menu.
        """)
        Claimchoices = gr.Dropdown(["An apparatus", "A method of use", "A method", "A method of manufacturing", "A system"], label='Choose Claim Type Here')        
        
        with gr.Row(scale=1, min_width=600):
            text1 = gr.Textbox(label="Input",
                              placeholder='Type in your idea here!', lines=5)
            text2 = gr.Textbox(label="Output", lines=5)
        with gr.Row():
            btn = gr.Button("Submit")
        with gr.Row():
          with gr.Accordion("Parameters"):
              temp = gr.Slider(minimum=0, maximum=1, value=0.6, label="Temperature", step=0.1)
              top_p = gr.Slider(minimum=0.5, maximum=1, value=0.95, label="Top P", step=0.1)
              tokens = gr.Slider(minimum=5, maximum=2058, value=512, label="Max Tokens", step=1)
                
        btn.click(fn=claim_selector, inputs=[text1, Claimchoices]).then(run_model, inputs=[text1, Claimchoices, temp, top_p, tokens], outputs=text2)
   
    with gr.Tab("Description Generator"):
        gr.Markdown(""" 
        Use this tool to expand your patent claim into a description. You can also use this tool to generate abstracts and give you ideas about the benefit of an invention by changing the settings in the dropdown menu.
         """)
        Descriptionchoices = gr.Dropdown(["Generate a Detailed Description Paragraph", "Generate a Abstract", "What are the Benefits/Technical Effects"], label='Choose Generation Type Here')
        with gr.Row(scale=1, min_width=600):
                    
            text1 = gr.Textbox(label="Input",
                              placeholder='Type in your idea here!', lines=5)
            text2 = gr.Textbox(label="Output", lines=5)
        with gr.Row():
            btn = gr.Button("Submit")
        with gr.Row():
          with gr.Accordion("Parameters"):
              temp = gr.Slider(minimum=0, maximum=1, value=0.6, label="Temperature", step=0.1)
              top_p = gr.Slider(minimum=0.5, maximum=1, value=0.95, label="Top P", step=0.1)
              tokens = gr.Slider(minimum=5, maximum=2058, value=512, label="Max Tokens", step=1)

        btn.click(fn=desc_selector, inputs=[text1, Descriptionchoices]).then(run_model, inputs=[text1, Descriptionchoices, temp, top_p, tokens], outputs=text2)
    
    # with gr.Tab("Prosecution Beta"):
    #     gr.Markdown(""" 
    #     Use this tool to generate ideas for how to overcome objections to novelty and inventive step. For now, this tool only works on relatively short inputs, so maybe try very simple inventions or short paragraphs.
    #      """)
    #     with gr.Row(scale=1, min_width=600):
    #         with gr.Column():
    #           application = gr.Text(label="Present Invention", lines=5)
    #           priorart = gr.Text(label="Prior Art Document", lines=5)        
    #         with gr.Column():
    #           text2 = gr.Textbox(label="Output", lines=5)
    #     with gr.Row():
    #         btn = gr.Button("Submit")
            
    #     with gr.Row():
    #       with gr.Accordion("Parameters"):
    #           temp = gr.Slider(minimum=0, maximum=1, value=0.6, label="Temperature", step=0.1)
    #           top_p = gr.Slider(minimum=0.5, maximum=1, value=0.95, label="Top P", step=0.1)
    #           tokens = gr.Slider(minimum=5, maximum=2058, value=512, label="Max Tokens", step=1)
        
    #     btn.click(fn=prosecute, inputs=[application, priorart, temp, top_p, tokens], outputs=text2)
     
    with gr.Tab("CPC Search Tool"):
        gr.Markdown("""
        Use this tool to classify your invention according to the Cooperative Patent Classification system. 
        Click on the link to initiate either an Espacenet or Google Patents classification search using the generated classifications. You can specify which you would like using the dropdown menu. 
        """)

        ClassifyChoices = gr.Dropdown(["Google Patent Search", "Espacenet Patent Search"], label='Choose Search Type Here')        
        with gr.Row(scale=1, min_width=600):
            with gr.Column(scale=5):
              userin = gr.Textbox(label="Input", placeholder='Type in your Claim/Description/Abstract Here',lines=5)
            with gr.Column(scale=1):
              with gr.Accordion("CPC classes"):
                output = gr.Markdown() #gr.Textbox(label="Output", lines=5)
        with gr.Row():
            classify_btn = gr.Button("Classify")
            classify_btn.click(fn=classifier, inputs=[userin, ClassifyChoices] , outputs=output)
        
    with gr.Tab("Chat"):
        gr.Markdown(""" 
        Do you want a bit more freedom over the outputs you generate? No problem! You can use a chatbot version of our model below. You can ask it anything. 
        We haven't done any filtering, so that we can understand exactly which biases/inappropriate responses exist in our model. 
        If you're concerned about any outputs, please get in contact with us to let us know what you saw. We will use this inform the development of later versions of this model.
         """)
        with gr.Row(scale=1, min_width=600):
            with gr.Column():
              userin = gr.Text(label="Question", lines=5)
            with gr.Column():
              text2 = gr.Textbox(label="Answer", lines=5)
        with gr.Row():
            btn = gr.Button("Submit")
            
        with gr.Row():
          with gr.Accordion("Parameters"):
              temp = gr.Slider(minimum=0, maximum=1, value=0.6, label="Temperature", step=0.1)
              top_p = gr.Slider(minimum=0.5, maximum=1, value=0.95, label="Top P", step=0.1)
              tokens = gr.Slider(minimum=5, maximum=2058, value=512, label="Max Tokens", step=1)
        btn.click(fn=Chat, inputs=[userin, temp, top_p, tokens], outputs=text2)

#    gr.Markdown(""" 
#    # THE CHATBOT
#    Do you want a bit more freedom over the outputs you generate? No problem! You can use a chatbot version of our model below. You can ask it anything. 
#    If you're concerned about a particular output, please 
#    """)
   
#    chatbot = gr.Chatbot([], elem_id="Claimed Assistant").style(height=500)
#    with gr.Row():
#        with gr.Column(scale=1):
#            txt = gr.Textbox(
#                show_label=False,
#                placeholder="Enter text and submit",
#            ).style(container=False)
#    
#    with gr.Row():
#        with gr.Accordion("Parameters"):
#            temp = gr.Slider(minimum=0, maximum=1, value=0.6, label="Temperature", step=0.1)
#            top_p = gr.Slider(minimum=0.5, maximum=1, value=0.95, label="Top P", step=0.1)
#            tokens = gr.Slider(minimum=5, maximum=1024, value=256, label="Max Tokens", step=1)
#        
#    txt.submit(add_text, [chatbot, txt], [chatbot, txt]).then(
#            generateresponse, [chatbot, temp, top_p, tokens], chatbot)

    gr.Markdown("""
    # HAVE AN IDEA? GET IT CLAIMED 
    
    In the future, we are looking to expand our model's capabilities further to assist in a range of IP related tasks.
    
    If you are interested in using a more powerful model that we have trained, or if you have any suggestions of features you would like to see us add, please get in touch!
        
    """)
demo.queue(max_size=20)
demo.launch(show_api=False)

