from argparse import ArgumentParser
from json import load

def parse_args():
    parser = ArgumentParser()
    parser.add_argument('input', type=str, nargs='+', \
                        help='Specify paths to files (e.g., path/to/*.json)')

    return parser.parse_args()


def json_to_markdown(filename):
    json = load(open(filename))

    markdown = f'# Dataset Card for {json["name"]}\n\n'

    markdown += f'You can find the '

    markdown += json['summary'] + '\n\n'

    for key in json:
        if key not in ('name', 'summary', 'sections'):
            markdown += f'#### {key}\n{json[key]}\n\n'

    markdown += '\n'.join(section_to_markdown(section) \
                          for section in json['sections'])

    with open(f'{filename[:-5]}.md', 'w') as f:
        f.write(markdown)


def section_to_markdown(section):
    markdown = f'{"#" * section["level"]} {section["title"]}\n\n'
    markdown += '\n'.join(subsection_to_markdown(subsection) \
                          for subsection in section['subsections'])

    return markdown + '\n'


def subsection_to_markdown(subsection):
    markdown = f'{"#" * subsection["level"]} {subsection["title"]}\n\n'
    markdown += '\n'.join(field_to_markdown(field) \
                          for field in subsection['fields'])

    return markdown + '\n'


def field_to_markdown(field):
    markdown = f'{"#" * field["level"]} {field["title"]}\n\n'

    if 'flags' in field and 'quick' in field['flags']:
        markdown += f'<!-- quick -->\n'

    if field.get('info', False):
        markdown += f'<!-- info: {field["info"]} -->\n'

    if field.get('scope', False):
        markdown += f'<!-- scope: {field["scope"]} -->\n'

    markdown += field.get('content', '')

    return markdown + '\n'


def main():
    """Converts JSON output from `reformat_json.py`
       to Markdown input for Data Cards Labs."""
    args = parse_args()
    for filename in args.input:
        if filename[-5:] == '.json':
            json_to_markdown(filename)


if __name__ == '__main__':
    main()