import os
from typing import List
from urllib.parse import urljoin

from settings import (
    DEFAULT_HUGGINGFACE_MODELS_API_BASE_URL,
    STATIC_PATH,
)

def masked(value: str, n_shown: int, length: int = None) -> str:
    """Returns a string with the first and last n_shown characters
    and the middle of the string replaced with '*'

    Args:
        value (str): The string to mask
        n_shown (int): The number of characters to show at the beginning and end of the string
        length (int, optional): The length of the string. If not given, it will be calculated as the length of the value. Defaults to None.

    Returns:
        str: The masked string
    """
    l = length or len(value)
    return value[0:n_shown] + '*'*(length-2*n_shown) + value[-n_shown:]


def ofuscated(value: str) -> str:
    """Returns a string with the first and last 4 characters
    and the middle of the string replaced with '*'

    Args:
        value (str): The string to mask

    Returns:
        str: The masked string
    """
    return masked(value, 4, len(value)//2)


def preview(label:str, value: str, ofuscate=False):
    """Print the variable name and its value in a nice way.
    If ofuscate is True, it will ofuscate the value

    Args:
        variable_name (str): The name of the variable to print
        ofuscate (bool, optional): If True, it will ofuscate the value. Defaults to False.
    """
    str_value = ofuscated(str(value)) if ofuscate else str(value)
    print(f"{label} = {str_value}")

def get_url_from_env_or_default_path(env_name: str, api_path: str) -> str:
    """Takes an url from the env variable (given the env name)
    or combines with urljoin the default models base url
    with the default path (given the path name)

    Args:
        env_name (str): The name of the environment variable to check
        api_path (str): The default path to use if the environment variable is not set

    Returns:
        str: The url to use
    """
    return os.environ.get(env_name) or urljoin(
        DEFAULT_HUGGINGFACE_MODELS_API_BASE_URL, api_path
    )

def get_file_as_string(file_name, path=STATIC_PATH) -> str:
    """Loads the content of a file given its name
    and returns all of its lines as a single string
    if a file path is given, it will be used
    instead of the default static path (from settings)

    Args:
        file_name (_type_): The name of the file to load.
        path (str, optional): The path to the file. Defaults to the current directory.

    Returns:
        str: The content of the file as a single string
    """
    with open(os.path.join(path, file_name), mode='r', encoding='UTF-8') as f:
        return f.read()


def get_sections(string: str, delimiter: str, up_to: int = None) -> List[str]:
    """Splits a string into sections given a delimiter

    Args:
        string (str): The string to split
        delimiter (str): The delimiter to use
        up_to (int, optional): The maximum number of sections to return.
                Defaults to None (which means all sections)

    Returns:
        List[str]: The list of sections (up to the given limit, if any provided)
    """
    return [section.strip()
            for section in string.split(delimiter)
            if (section and not section.isspace())][:up_to]
