import streamlit as st
import pinecone
from sentence_transformers import SentenceTransformer
from concurrent.futures import ThreadPoolExecutor, as_completed
import logging
import requests

PINECONE_KEYS = {
    1: {"key": st.secrets["PINECONE_KEY"], "env": 'us-central1-gcp'},
    2: {"key": st.secrets["PINECONE_KEY_2"], "env": 'asia-southeast1-gcp-free'},
    3: {"key": st.secrets["PINECONE_KEY_3"], "env": 'us-west1-gcp-free'},
    4: {"key": st.secrets["PINECONE_KEY_4"], "env": 'asia-southeast1-gcp-free'},
    5: {"key": st.secrets["PINECONE_KEY_5"], "env": 'gcp-starter'},
    6: {"key": st.secrets["PINECONE_KEY_6"], "env": 'us-west1-gcp-free'},
    7: {"key": st.secrets["PINECONE_KEY_7"], "env": 'gcp-starter'},
    8: {"key": st.secrets["PINECONE_KEY_8"], "env": 'gcp-starter'},
    9: {"key": st.secrets["PINECONE_KEY_9"], "env": 'gcp-starter'},
    #10: {"key": st.secrets["PINECONE_KEY_10"], "env": 'gcp-starter'}
    11: {"key": st.secrets["PINECONE_KEY_11"], "env": 'gcp-starter'},
    12: {"key": st.secrets["PINECONE_KEY_12"], "env": 'gcp-starter'},
    13: {"key": st.secrets["PINECONE_KEY_13"], "env": 'gcp-starter'},
    14: {"key": st.secrets["PINECONE_KEY_14"], "env": 'gcp-starter'},
    15: {"key": st.secrets["PINECONE_KEY_15"], "env": 'gcp-starter'},
    16: {"key": st.secrets["PINECONE_KEY_16"], "env": 'gcp-starter'},
    17: {"key": st.secrets["PINECONE_KEY_17"], "env": 'gcp-starter'},
}

INDEX_ID = 'youtube-search'

with open("styles.css") as f:
    st.markdown(f'<style>{f.read()}</style>', unsafe_allow_html=True)


@st.experimental_singleton
def init_pinecone(key, index_id, environment):
    pinecone.init(api_key=key, environment=environment)
    return pinecone.Index(index_id)


@st.experimental_singleton
def init_retriever():
    return SentenceTransformer("multi-qa-mpnet-base-dot-v1")


def initialize_pinecone_index(i, details):
    pinecone.init(api_key=details['key'], environment=details['env'])
    setattr(st.session_state, f'index{i}', pinecone.Index(INDEX_ID))


def query_single_index(index, xq, channels, top_k=100, include_values=True, include_metadata=True):
    return index.query(
        xq,
        top_k=top_k,
        include_values=include_values,
        include_metadata=include_metadata,
        filter={'channel_name': {'$in': channels}}
    )


def make_query(query, retriever, channels, top_k=100, include_values=True, include_metadata=True):
    xq = retriever.encode([query]).tolist()
    logging.info(f"Query: {query}")
    all_matches = []
    
    with ThreadPoolExecutor() as executor:
        futures = {}
        for i, details in PINECONE_KEYS.items():
            index_name = f'index{i}'
            index = getattr(st.session_state, index_name, None)
            if not index:
                initialize_pinecone_index(i, details)
                index = getattr(st.session_state, index_name)
            futures[executor.submit(query_single_index, index, xq, channels, top_k, include_values, include_metadata)] = i

        for future in as_completed(futures):
            i = futures[future]
            try:
                matches = future.result()
                all_matches += matches['matches']
            except Exception as e:
                logging.error(f"Query failed for index {i}: {e}")
                # Re-initialize Pinecone index if needed
                initialize_pinecone_index(i, PINECONE_KEYS[i])
                
    # Sort matches by score in descending order
    all_matches.sort(key=lambda x: x['score'], reverse=True)
    logging.info(f"Found {len(all_matches)} matches.")
    
    # Return top_k matches
    return all_matches[:top_k]


for idx, details in PINECONE_KEYS.items():
    setattr(st.session_state, f'index{idx}', init_pinecone(details['key'], INDEX_ID, details['env']))

retriever = init_retriever()


def card(thumbnail: str, title: str, urls: list, contexts: list, starts: list, ends: list,  max_contexts=3):
    # Limit the contexts, urls, starts, and ends to max_contexts
    contexts = contexts[:max_contexts]
    urls = urls[:max_contexts]
    starts = starts[:max_contexts]
    ends = ends[:max_contexts]
    
    meta = [(e, s, u, c) for e, s, u, c in zip(ends, starts, urls, contexts)]
    meta.sort(reverse=False)
    text_content = []
    current_start = 0
    current_end = 0
    for end, start, url, context in meta:
        # reformat seconds to timestamp
        time = start / 60
        mins = f"0{int(time)}"[-2:]
        secs = f"0{int(round((time - int(mins)) * 60, 0))}"[-2:]
        timestamp = f"{mins}:{secs}"
        if start < current_end and start > current_start:
            # this means it is a continuation of the previous sentence
            text_content[-1][0] = text_content[-1][0].split(context[:10])[0]
            text_content.append([f"[{timestamp}] {context.capitalize()}", url])
        else:
            text_content.append(["xxLINEBREAKxx", ""])
            text_content.append([f"[{timestamp}] {context}", url])
        current_start = start
        current_end = end
    html_text = ""
    for text, url in text_content:
        if text == "xxLINEBREAKxx":
            html_text += "<br>"
        else:
            html_text += f"<small><a href={url}>{text.strip()}... </a></small>"
        html = f"""
            <div class="container-fluid">
                <div class="row align-items-start">
                    <div class="col-md-4 col-sm-4">
                        <div class="position-relative">
                             <a href={urls[0]}><img src={thumbnail} class="img-fluid" style="width: 192px; height: 106px"></a>
                        </div>
                    </div>
                    <div  class="col-md-8 col-sm-8">
                        <h2>{title}</h2>
                    </div>
                <div>
                    {html_text}
            <br><br>
            """
    return st.markdown(html, unsafe_allow_html=True)

st.write("""
# YouTube Q&A
""")

st.info("""
For optimal results, please phrase your query as a question and ensure correct spelling. Results are ranked by relevance.
""")


st.markdown("""
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.0.0/dist/css/bootstrap.min.css" integrity="sha384-Gn5384xqQ1aoWXA+058RXPxPg6fy4IWvTNh0E263XmFcJlSAwiGgFAW/dAiS6JXm" crossorigin="anonymous">
""", unsafe_allow_html=True)

query = st.text_input("Search!", "")

with st.expander("Advanced Options"):
    channel_options = st.multiselect(
        'Channels to Search',
        ['Art Katz', 'A. W. Tozer', 'Dean Braxton', 'Derek Prince', 'Bruce D. Allen', 'Bill Wiese', 'Charles Spurgeon', 'Cecil Pearson', 'Conrad Rocks!', 'D.W. Pastor Barry', 'David Wilkerson', 'Francis Chan', 'Fellowship of the Martyrs',
         'George Whitefield', 'Glynda Lomax', 'Henry Gruver', 'HolyLandSite', 'James Maloney', 'Jesus My King Church', 'JOHN SCOTLAND', 'Jonathan Edwards', 'Kevin Zadai', 'Leonard Ravenhill', 'MIchael Van Vlymen', 'Multiplying Disciples',
         'Monica Dennington', 'Neville Johnson', 'Nick Vujicic', 'Rev. Elation', 'Ron Wyatt', 'Sadhu Sundar Selvaraj', 'ShekinahWorshipTV', 'SuperHouseChurch', 'Tony Kemp', 'Torch of Christ Ministries', 
         'Voice of the Martyrs USA', 'Walk After Christ', 'WATCHERS OF TRUTH', 'Zac Poonen'],
        ['Art Katz', 'A. W. Tozer', 'Dean Braxton', 'Derek Prince', 'Bruce D. Allen', 'Bill Wiese', 'Charles Spurgeon', 'Cecil Pearson', 'Conrad Rocks!', 'D.W. Pastor Barry', 'David Wilkerson', 'Francis Chan', 'Fellowship of the Martyrs',
         'George Whitefield', 'Glynda Lomax', 'Henry Gruver', 'HolyLandSite', 'James Maloney', 'Jesus My King Church', 'JOHN SCOTLAND', 'Jonathan Edwards', 'Kevin Zadai', 'Leonard Ravenhill', 'MIchael Van Vlymen', 'Multiplying Disciples',
         'Monica Dennington', 'Neville Johnson', 'Nick Vujicic', 'Rev. Elation', 'Ron Wyatt', 'Sadhu Sundar Selvaraj', 'ShekinahWorshipTV', 'SuperHouseChurch', 'Tony Kemp', 'Torch of Christ Ministries', 
         'Voice of the Martyrs USA', 'Walk After Christ', 'WATCHERS OF TRUTH', 'Zac Poonen']
    )

if query != "":
    matches = make_query(
        query, retriever, channel_options, top_k=100
    )

    results = {}
    order = []
    for context in matches:
        video_id = context['metadata']['url'].split('/')[-1]
        if video_id not in results:
            results[video_id] = {
                'title': context['metadata']['title'],
                'urls': [f"{context['metadata']['url']}?t={int(context['metadata']['start'])}"],
                'thumbnail_urls': [context['metadata']['thumbnail_url']],
                'contexts': [context['metadata']['text']],
                'starts': [int(context['metadata']['start'])],
                'ends': [int(context['metadata']['end'])]
            }
            order.append(video_id)
        else:
            results[video_id]['urls'].append(
                f"{context['metadata']['url']}?t={int(context['metadata']['start'])}"
            )
            results[video_id]['thumbnail_urls'].append(context['metadata']['thumbnail_url'])
            results[video_id]['contexts'].append(
                context['metadata']['text']
            )
            results[video_id]['starts'].append(int(context['metadata']['start']))
            results[video_id]['ends'].append(int(context['metadata']['end']))
    # now display cards
    for video_id in order:
        card(
            thumbnail=results[video_id]['thumbnail_urls'][0],
            title=results[video_id]['title'],
            urls=results[video_id]['urls'],
            contexts=results[video_id]['contexts'],
            starts=results[video_id]['starts'],
            ends=results[video_id]['ends'],
            max_contexts=3
        )