# Plugin will run on this file & Strictly only contains codes for routing between files!
from flask import Flask, render_template, request, flash, g
from TextSummarizationFeature import summarize, getTextSumContents, insertTextSumRow
from BreakdownFeature import breakdown, getBreakdownContents, insertBreakdownRow
from TranslationFeature import translate_text, switch, getTranslatedContents, insertTranslationRow
from GenerationFeature import generate, getTextGenContents, insertTextGenRow

app = Flask(__name__)
app.secret_key = 'refineverseAdmin' # Used to encrypt cookies & sessions

# Routing to Main Dashboard/homepage file
@app.route('/')
def index():
    return render_template('RefineverseDashboardUI.html')

# Routing to text summarization file
@app.route('/text_summarization', methods=["POST", "GET"])
def text_summarization():
    if request.method == "POST":
        try:
            # Grab the user story text from the textarea in html form
            Entered_story = request.form["input_text"]
            
            # The results are stored into a dictionary variable
            summarizedStory = summarize(Entered_story)
            
            flash("Your user story has been summarized!") # Displays a success message using flash, which is part of the Flask framework
            
            # Insert into TextSummarization table in Refineverse.db
            insertTextSumRow(Entered_story, summarizedStory)
            
            # Render and display summarized user story
            return render_template('TextSummarizationUI.html', summarizedStory=summarizedStory)
        
        # Exception handling messages for specific errors
        except ValueError as e:
            if str(e) == "Empty input!":
                flash("The input text cannot be empty! Please enter a user story.", 'error')
                return render_template('TextSummarizationUI.html')
            elif str(e) == "Incorrect format!":
                flash("Incorrect user story format! Please enter in the right format.", 'error')
                return render_template('TextSummarizationUI.html')
            elif str(e) == "Invalid length!":
                flash("Your inputted user story is too short to summarize. Please enter a longer story!", 'error')
                return render_template('TextSummarizationUI.html')
            else: # As a final resort, simply print out the error name
                flash("An error of type '{}' occurred: {}".format(type(e).__name__, str(e)), 'error')
                return render_template('TextSummarizationUI.html')
        
        except KeyError:
            flash("Please enter a valid user story!")
            return render_template('TextSummarizationUI.html')
        
        # Catch-all exception handling
        except Exception as e:
            flash("An error of type '{}' occurred: {}".format(type(e).__name__, str(e)), 'error')
            return render_template('TextSummarizationUI.html')
    
    else:
        return render_template('TextSummarizationUI.html')

# Routing to summarization table file
@app.route('/summarization_table')
def summarization_table():
    # Get the summarization data from the database
    summarizations = getTextSumContents()
    
    # Render the summarization data as an HTML table
    return render_template('SummarizationTable.html', summarizations=summarizations)

# Routing to Project Task Breakdown file
@app.route("/project_breakdown", methods=["POST", "GET"]) # This tells flask the route to get to the page
def project_breakdown():
    if request.method == "POST": # POST occurs when submitting a form, as specified in the HTML file
        try:
            # Grab the user story contents
            userStory = request.form["user-story-text"]
            
            # The results are stored into a dictionary variable
            processedLabel = breakdown(userStory)
            
            # Display success popup message
            flash("Your user story has been allocated as a " + processedLabel + " task!")
            
            insertBreakdownRow(userStory, processedLabel) # Inserts data into the Breakdown table
            rows = getBreakdownContents() # Grab all contents inside the Breakdown table
            
            return render_template('ProjectBreakdownUI.html', rows=rows)
        
        # Exception handling messages for specific errors
        except KeyError:
            flash("Please enter a valid user story!", 'error')
            rows = getBreakdownContents()
            return render_template('ProjectBreakdownUI.html', row=rows)
        
        # Catch-all exception handling
        except Exception as e:
            flash("An error of type '{}' occurred: {}".format(type(e).__name__, str(e)), 'error')
            rows = getBreakdownContents()
            return render_template('ProjectBreakdownUI.html', rows=rows)
        
    else: # For "GET" scenarios (loading the page, etc.)
        rows = getBreakdownContents()  # To always display the table, we must grab the contents of Breakdown every time the page loads
        return render_template('ProjectBreakdownUI.html', rows=rows)

# Routing to Translation file
@app.route('/language_translation', methods=["POST", "GET"])
def language_translation():
    if request.method == "POST":
        try:
            # Grab all relevant information for processing
            input_text = request.form['input'] # Grab user text input
            
            # Grab source language code
            source_language = request.form['source_language']
            
            # Grab target language code
            target_language = request.form['target_language']
            
            # Generate translated text using custom translation function
            translatedStory = translate_text(input_text, source_language, target_language)
            
            # Display success popup message
            flash("Your user story has been translated to " + switch(target_language) + "  !")
            
            # Insert into Translation table in Refineverse.db
            insertTranslationRow(input_text, translatedStory)
            
            # Display the page
            return render_template('LanguageTranslationUI.html', input_text=input_text, translatedStory=translatedStory)
        
        # Exception handling messages for specific errors
        except ValueError as e:
            if str(e) == "Empty input!":
                flash("The input text cannot be empty! Please enter a user story.", 'error')
                return render_template('LanguageTranslationUI.html')
            elif str(e) == "Incorrect format!":
                flash("Unable to translate your user story. Please enter in the correct format.", 'error')
                return render_template('LanguageTranslationUI.html')
            else: # As a final resort, simply print out the error name
                flash("An error of type '{}' occurred: {}".format(type(e).__name__, str(e)), 'error')
                return render_template('LanguageTranslationUI.html')
        
        # Catch-all exception handling
        except Exception as e:
            flash("An error of type '{}' occurred: {}".format(type(e).__name__, str(e)), 'error')
            return render_template('LanguageTranslationUI.html')
    
    else:
        return render_template('LanguageTranslationUI.html') 

# Routing to translation table file
@app.route('/translation_table')
def translation_data():
    # Get the translation data from the database
    translations = getTranslatedContents()
    
    # Render the translation data as an HTML table
    return render_template('TranslationTable.html', translations=translations)

# Routing to text summarization file
@app.route('/text_generation', methods=["POST", "GET"])
def text_generation():
    if request.method == "POST":
        try:
            # Grab the user story text from the textarea in html form
            Entered_story = request.form["input_text"]
            
            # The results are stored into a dictionary variable
            generatedStory = generate(Entered_story)
            
            # Display a success message for the user
            flash("Your user story has been generated!")
            
            # Insert into TextGeneration table in Refineverse.db
            insertTextGenRow(Entered_story, generatedStory)
            
            # Render and display summarized user story
            return render_template('TextGenerationUI.html', generatedStory=generatedStory)
        
        # Exception handling messages for specific errors
        except ValueError as e:
            if str(e) == "Empty input!":
                flash("The input text cannot be empty! Please enter a user story.", 'error')
                return render_template('TextGenerationUI.html')
            elif str(e) == "Incorrect format!":
                flash("Incorrect user story format! Please enter in the right format.", 'error')
                return render_template('TextGenerationUI.html')
            else: # As a final resort, simply print out the error name
                flash("An error of type '{}' occurred: {}".format(type(e).__name__, str(e)), 'error')
                return render_template('TextGenerationUI.html')
        
        except KeyError:
            flash("Please enter a valid user story!")
            return render_template('TextGenerationUI.html')
        
        # Catch-all exception handling
        except Exception as e:
            flash("An error of type '{}' occurred: {}".format(type(e).__name__, str(e)), 'error')
            return render_template('TextGenerationUI.html')
    
    else:
        return render_template('TextGenerationUI.html')

# Routing to generation table file
@app.route('/generation_table')
def generation_table():
    # Get the generation data from the database
    generations = getTextGenContents()
    
    # Render the generation data as an HTML table
    return render_template('GenerationTable.html', generations=generations)

# Used when the application is torn down
# Its purpose is to close the database connection if it has not been closed
@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close() # Closes the database connection

# Initialise the app
if __name__ == '__main__':
    app.run(host="0.0.0.0", port=7860) # For HF hosting
    #app.run(debug=False) # can set to True/False for local testing