import os
from jinja2 import Environment, FileSystemLoader, select_autoescape
from get_paperinfo_fromurls import get_paperinfo_fromurls
import gradio as gr

class CARDS_TEMPLATE(object):
    def __init__(self, path_to_template, template_filename):
        self.path_to_template = path_to_template
        self.template_filename = template_filename
        self.template = self._get_template()
        self.rendered_html = None

    def _get_template(self):
        env = Environment(
                    autoescape=select_autoescape(
                        enabled_extensions=('html'),
                        default_for_string=True,
                    ),
                    loader=FileSystemLoader(self.path_to_template)
                )
        return env.get_template(self.template_filename)

    def render(self, paper_details_iterator):
        self.rendered_html = self.template.render(paper_details=paper_details_iterator)

    def save_html(self, output_dir=None, output_htmlfile=None):
        with open(os.path.join(output_dir, output_htmlfile), "w") as f:
            f.write(self.rendered_html)

template_file = "htmlcard.html"
template_path = ""
card_template = CARDS_TEMPLATE(
                path_to_template = template_path,
                template_filename = template_file,
                )

CSS = """
#url-textbox {
    padding: 0 !important;
    font-size: 16px;
}

.gradio-container {
    background-color: transparent;
}

.gradio-container .gr-button-primary {
    background: #b31b1b;
    border: 1px solid #b31b1b;
    border-radius: 8px;
    color: white;
    font-weight: bold;
    font-size: 16px;
}

#ctr {
    text-align: center;
}

#htel {
    justify-content: center;
    text-align: center;
}
"""

examples = [
    [
        "https://arxiv.org/abs/2208.14178v1",
    ]
]

def create_html_card(arxiv_link):
    paper_details = get_paperinfo_fromurls(arxiv_link)
    card_template.render(paper_details_iterator=paper_details)
    return card_template.rendered_html

demo = gr.Blocks(css=CSS)
with demo:
    with gr.Column():
        gr.Markdown("# arXiv Cards Generator ⚙️", elem_id="ctr")
        gr.Markdown(
            """
            Need a simple and visual way to share arXiv papers on presentations, blogposts, messages?  
            This gradio demo allows for creating arXiv cards including arXiv identifier, title, authors, abstract

            Simply paste the url link of the arXiv paper and generate!
            """
        )

    with gr.Column():
        with gr.Row():
            text = gr.Textbox(
                show_label=False,
                placeholder="Paste arXiv link (abs of pdf)",
                lines=1,
                max_lines=1,
                elem_id="url-textbox",
            )
            button = gr.Button("Generate", variant="primary")
        with gr.Row():
            card = gr.HTML(elem_id="htel")
        with gr.Row():
            gr.Examples(
                examples=examples,
                inputs=[text],
            )

    with gr.Column():
        gr.Markdown("### Resources and inspirations", elem_id="ctr")
        gr.Markdown(
            """
            - The code for retrieving the information using arXiv API is mainly taken from [github.com/kunalghosh/Conference-Grok](https://github.com/kunalghosh/Conference-Grok).
            - The [pdf2preview](https://huggingface.co/spaces/chuanenlin/pdf2preview) space is also a great way to share academic publications on slides.
            
            **Author**: [eliolio](https://huggingface.co/eliolio)
            """)
    button.click(
        fn=create_html_card,
        inputs=[text],
        outputs=[card]
    )



if __name__ == "__main__":
    demo.launch()