import streamlit as st
import numpy as np
from tensorflow.keras.preprocessing.sequence import pad_sequences
import nltk
from nltk.sentiment import SentimentIntensityAnalyzer
from tensorflow.keras.preprocessing.text import Tokenizer
from tensorflow.keras.models import load_model
import json

nltk.download('vader_lexicon')


# Load the model and tokenizer
model_path = 'model.h5'
tokenizer_path = 'tokenizer.json'

# Load the tokenizer
with open(tokenizer_path, 'r') as tokenizer_file:
    word_index = json.load(tokenizer_file)
    tokenizer = Tokenizer(num_words=100000)
    tokenizer.word_index = word_index

classes = ['ADHD', 'OCD', 'aspergers', 'depression', 'ptsd']

# Disease definitions
disease_definitions = {
    'ADHD': "Attention-Deficit/Hyperactivity Disorder (ADHD) is a neurodevelopmental disorder characterized by persistent patterns of inattention, hyperactivity, and impulsivity.",
    'OCD': "Obsessive-Compulsive Disorder (OCD) is a mental health disorder characterized by recurring unwanted thoughts (obsessions) and repetitive behaviors or mental rituals (compulsions).",
    'aspergers': "Asperger's Syndrome, also known as Autism Spectrum Disorder (ASD), is a developmental disorder characterized by difficulties in social interaction, repetitive patterns of behavior, and limited interests or activities.",
    'depression': "Depression is a common mental health disorder characterized by persistent sadness, loss of interest or pleasure in activities, changes in appetite or sleep patterns, and feelings of worthlessness or guilt.",
    'ptsd': "Post-Traumatic Stress Disorder (PTSD) is a mental health condition triggered by a traumatic event. Symptoms may include flashbacks, nightmares, severe anxiety, and uncontrollable thoughts about the event."
}

# Coping strategies
coping_strategies = {
    'ADHD': 'Try breaking tasks into smaller, manageable steps and using reminders or organizational tools.',
    'OCD': 'Practice exposure and response prevention (ERP) techniques and consider therapy options like Cognitive Behavioral Therapy (CBT).',
    'aspergers': 'Focus on building social skills through therapy and practice. Seek support from autism support groups.',
    'depression': 'Reach out to a mental health professional for therapy or medication options. Engage in self-care activities and reach out to supportive friends or family.',
    'ptsd': 'Consider therapy options such as Cognitive Processing Therapy (CPT) or Eye Movement Desensitization and Reprocessing (EMDR). Practice grounding techniques during moments of distress.'
}

# ... (Rest of the code remains the same)
# Resources
resources = {
    'ADHD': 'Vidyasagar Institute of Mental Health and Neurosciences: http://www.vimhanshospital.org/',
    'OCD': 'Serenity Neuropsychiatry Clinic: http://serenityclinic.org/',
    'aspergers': 'Autism Society of India: http://www.autismsocietyofindia.org/',
    'depression': 'iCALL - Psychosocial Helpline: https://www.icallhelpline.org/',
    'ptsd': 'Mpower - The Centre: https://www.mpowerminds.com/'
}

# Doctors
doctors = {
    'ADHD': ['Dr. John Doe', 'Dr. Jane Smith'],
    'OCD': ['Dr. David Johnson', 'Dr. Emily Davis'],
    'aspergers': ['Dr. Michael Brown', 'Dr. Sarah Wilson'],
    'depression': ['Dr. Christopher Lee', 'Dr. Samantha Taylor'],
    'ptsd': ['Dr. Robert Martinez', 'Dr. Jessica Thompson']
}

# Podcasts
podcasts = {
    'ADHD': ['ADHD Experts Podcast', 'The ADHD Manual Podcast'],
    'OCD': ['The OCD Stories Podcast', 'The OCD & Anxiety Show'],
    'aspergers': ['The Autism Helper Podcast', 'Autism Spectrum Podcast'],
    'depression': ['The Depression Files Podcast', 'The Hilarious World of Depression'],
    'ptsd': ['The Trauma Therapist Podcast', 'PTSD Poetry Podcast']
}

# Books
books = {
    'ADHD': ['Driven to Distraction by Edward M. Hallowell', 'Taking Charge of Adult ADHD by Russell A. Barkley'],
    'OCD': ['Brain Lock by Jeffrey M. Schwartz', 'The OCD Workbook by Bruce M. Hyman'],
    'aspergers': ['The Complete Guide to Asperger\'s Syndrome by Tony Attwood', 'Uniquely Human by Barry M. Prizant'],
    'depression': ['The Noonday Demon by Andrew Solomon', 'Lost Connections by Johann Hari'],
    'ptsd': ['The Body Keeps the Score by Bessel van der Kolk', 'After the War Zone by Laurie B. Slone']
}

# Music
music = {
    'ADHD': ['Focus Playlist on Spotify', 'Energetic Beats Playlist on Apple Music'],
    'OCD': ['Relaxing Sounds Playlist on Spotify', 'Calm Vibes Playlist on Apple Music'],
    'aspergers': ['Soothing Melodies Playlist on Spotify', 'Peaceful Piano Playlist on Apple Music'],
    'depression': ['Mood Booster Playlist on Spotify', 'Feel-Good Hits Playlist on Apple Music'],
    'ptsd': ['Healing Sounds Playlist on Spotify', 'Emotional Support Playlist on Apple Music']
}

def get_response(user_input):
    #if 'hello' in user_input.lower() or 'hi' in user_input.lower():
        #return "Hello! How can I assist you today?"

    #if 'thank you' in user_input.lower() or 'thanks' in user_input.lower() or 'thanx' in user_input.lower():
        #return "You are welcome!"

    # Check if the user asks for the definition of a disease
    disease_keywords = ['definition', 'what is', 'explain', "what's", 'whats']
    for keyword in disease_keywords:
        if keyword in user_input.lower():
            for disease, definition in disease_definitions.items():
                if disease.lower() in user_input.lower():
                    return f"The definition of {disease} is: {definition}"

    sequence = tokenizer.texts_to_sequences([user_input])
    sequence = pad_sequences(sequence, maxlen=100)

    # Load the model
    model = load_model(model_path)

    prediction = model.predict(sequence)
    predicted_class_index = np.argmax(prediction)
    predicted_class = classes[predicted_class_index]

    # Calculate sentiment score
    sid = SentimentIntensityAnalyzer()
    sentiment_score = sid.polarity_scores(user_input)['compound']

    # Determine the topic based on the highest predicted probability
    predicted_topic = classes[np.argmax(prediction)]

    if predicted_topic in classes:
        # Provide coping strategies, resources, doctors, podcasts, books, and music based on the predicted topic
        if np.max(prediction) < 0.7:
            return f"The model predicts your mental health issue to be {predicted_class}, but it's recommended to consult a doctor for a proper diagnosis."

        response = f"Predicted Disease: {predicted_class}\n"

        if predicted_topic in coping_strategies:
            response += f"\nCoping Strategy: {coping_strategies[predicted_topic]}\n"

        if predicted_topic in resources:
            response += f"\nResource: {resources[predicted_topic]}\n"

        if predicted_topic in doctors:
            response += f"\nDoctors: {', '.join(doctors[predicted_topic])}\n"

        if predicted_topic in podcasts:
            response += f"\nPodcasts: {', '.join(podcasts[predicted_topic])}\n"

        if predicted_topic in books:
            response += f"\nBooks: {', '.join(books[predicted_topic])}\n"

        if predicted_topic in music:
            response += f"\nMusic: {', '.join(music[predicted_topic])}\n"

        return response

    if sentiment_score >= 0.2:
        return "I'm glad to hear that you're feeling happy! If you need any mental health support, feel free to share."
    elif sentiment_score <= -0.2:
        return "I'm sorry to hear that you're feeling down. Remember, you're not alone. Reach out to someone you trust or consider seeking professional help."

    return "Thank you for sharing. How can I assist you today?"

def main():
    st.title("Mental Health Chatbot")
    st.write("Enter your message and get a response from the chatbot:")

    user_input = st.text_input("User Input:")

    if st.button("Submit"):
        response = get_response(user_input)
        st.write("Bot Response:")
        st.write(response)

if __name__ == '__main__':
    main()

