### 1. Imports and class names setup ###
import os
import cv2
import dlib
import time
import numpy as np
import gradio as gr
from PIL import Image
from typing import Dict, Tuple
import torch
import torchvision
from torch import nn
from torchvision import transforms
from timeit import default_timer as timer
from model import create_ResNetb34_model

# Setup class names
class_names = ["Autistic", "Non_Autistic"]

### 2. Model and transforms preparation ###


resnet34, resnet34_transforms = create_ResNetb34_model(num_classes=len(class_names) )

# Load saved weights
resnet34.load_state_dict(torch.load(f="pretrained_resnet34_feature_extractor98.pth",
                                    map_location=torch.device("cpu"),))
### 3. Predict function ###
# Create predict function
# def predict(img)-> Tuple[Dict, float]:

#     # Start the timer
#     start_time=timer()

#     # Transform the target image and add a batch dimension
#     img=img.convert('RGB')
#     img = resnet34_transforms(img).unsqueeze(0)
#     # put model into evaluation mode and turn infarance mode
#     resnet34.eval()
#     with torch.inference_mode():

#         # Pass the transformed image through the model and turn the prediction logits into prediction probabilities
#         pred_probs=torch.softmax(resnet34(img),dim=1)
#     # Create a prediction label and prediction probability dictionary for each prediction class (this is the required format for Gradio's output parameter)

#     pred_labels_and_probs={class_names[i]:float(pred_probs[0][i]) for i in range(len(class_names))}
#     # Calculate the prediction time
#     pred_time = round(timer() - start_time, 5)

#     # Return the prediction dictionary and prediction time
#     return pred_labels_and_probs, pred_time
def predict_with_face_detection(img) :
    """
    Detects faces in an image, performs a prediction on the faces, and returns the prediction and time taken.

    Args:
        img_path: Path to the image.

    Returns:
        A tuple of the prediction dictionary and the prediction time.
    """
    #
    img_array = np.asarray(img)
    # Create a face detector.
    face_detector = dlib.get_frontal_face_detector()
    img=img.convert('RGB')

    # Detect faces in the image.
    faces = face_detector(img_array)

    # Open the target image.
    #img = Image.open(img_path)

    # Check if any faces were detected.
    if len(faces) >= 1:
        # Start the timer.
        start_time = time.perf_counter()

        # Create lists to store the predicted labels and probabilities for each face.
        predicted_labels = []
        predicted_probs = []

        # Loop through each detected face.
        for i, face in enumerate(faces):
            # Get the coordinates of the face bounding box.
            x1, y1, x2, y2 = face.left(), face.top(), face.right(), face.bottom()

            # Crop the face from the image.
            cropped_face = img.crop((x1, y1, x2, y2))

            # Transform the cropped face and add a batch dimension.
            transformed_face = resnet34_transforms(cropped_face).unsqueeze(0)

            # Put the model into evaluation mode and disable gradient calculation.
            resnet34.eval()
            with torch.inference_mode():
                # Pass the transformed face through the model and turn the prediction logits into probabilities.
                pred_probs = torch.softmax(resnet34(transformed_face), dim=1)

            # Get the predicted label with the highest probability.
            predicted_label = class_names[pred_probs.argmax()]

            # Get the corresponding probability score.
            predicted_prob = float(pred_probs.max())

            # Append the predicted label and probability to the lists.
            predicted_labels.append(predicted_label)
            predicted_probs.append(predicted_prob)

        # Calculate the prediction time.
        pred_time = round(time.perf_counter() - start_time, 5)

        # Create the output in Hugging Face format for handling multiple faces.
        output = {
            "labels": predicted_labels,
            "scores": predicted_probs
        }

        # If only one face is detected, return the final prediction label directly.
        if len(faces) == 1:
            return predicted_labels[0], pred_time
        else:
            # Otherwise, return the Hugging Face format for multiple faces.
            return output, pred_time
    else:
        # No face detected or multiple faces detected.
        return "Image Must Include At Least One Child's Face", 0.0


### 4. Gradio app ###
example_list = [["examples/" + example] for example in os.listdir("examples")]

# Create title, description and article strings
title = "ASD diagnosis"
description = """A feature extractor computer vision model to Identification of Autism in Children Using Static Facial Features and Deep Neural Networks.
                 
      Requirements ⇒ 
            image must be only child’s face.
            visible opened eyes (no hair blocking the eyes) and visible ears ."""


article = """Autism spectrum disorder (ASD) is a complicated neurological developmental disorder
that manifests itself in a variety of ways. The child diagnosed with ASD and their parents’ daily
lives can be dramatically improved with early diagnosis and appropriate medical intervention. The
applicability of static features extracted from autistic children’s face photographs as a biomarker to
distinguish them from typically developing children is investigated in this study paper. We used five
pre-trained CNN models: MobileNet, Xception, EfficientNetB0, EfficientNetB1, and EfficientNetB2 as
feature extractors and a DNN model as a binary classifier to identify autism in children accurately.
We used a publicly available dataset to train the suggested models, which consisted of face pictures
of children diagnosed with autism and controls classed as autistic and non-autistic. The Resnet34
model outperformed the others, with an AUC of 98.63%, a sensitivity of 88.46%, and an NPV of 88%.
EfficientNetB0 produced a consistent prediction score of 59% for autistic and non-autistic groups
with a 95% confidence level."""

# Create the Gradio demo
input_1= gr.inputs.Image(type='pil', label="upload Image", source="upload",optional=True)
input_2 = gr.inputs.Image(type='pil', label="take photo", source="webcam",optional=True)
# inputs= [input_1, input_2]
app1 = gr.Interface(fn=predict_with_face_detection, # mapping function from input to output
                    inputs=input_1 ,# what are the inputs?
                    outputs=[gr.Label(num_top_classes=3, label="Predictions"), # what are the outputs?
                             gr.Number(label="Prediction time (s)")], # our fn has two outputs, therefore we have two outputs
                    examples=example_list,
                    title=title,
                    description=description,
                    article=article)
app2=gr.Interface(fn=predict_with_face_detection, # mapping function from input to output
                    inputs=input_2 ,# what are the inputs?
                    outputs=[gr.Label(num_top_classes=3, label="Predictions"), # what are the outputs?
                             gr.Number(label="Prediction time (s)")], # our fn has two outputs, therefore we have two outputs
                    examples=example_list,
                    title=title,
                    description=description,
                    article=article)
demo = gr.TabbedInterface([app1, app2], ["upload photo", "take photo"])

# Launch the demo!
demo.launch()
