import streamlit as st
import pandas as pd
import numpy as np
from buy_or_rent import Buy_or_Rent_Model
from scipy.stats import norm, skew
from utils.general import calculate_percentiles
from utils.streamlit_utils import sticky_bottom_bar
from plot import plot_hist_from_list
import hydralit_components as hc
import warnings

warnings.simplefilter(action="ignore", category=FutureWarning)


def generate_main_body(
    model: Buy_or_Rent_Model,
    mortgage_interest_annual_list=np.array([0.05]),
    property_price_growth_annual_list=np.array([0.026]),
    rent_increase_list=np.array([0.01325]),
    investment_return_annual_list=np.array([0.06]),
    years_until_sell_list=np.array([20]),
):
    
    adjust_for_inflation_bool = st.sidebar.toggle("Adjust for inflation (2% a year)")
    # use_present_value = st.toggle('Use present value instead of future value')
    # define what option labels and icons to display
    option_data = [
        {"icon": "bi bi-calculator", "label": "Typical Outcome"},
        {"icon": "bi bi-bar-chart-line", "label": "Simulation Results"},
    ]

    # override the theme, else it will use the Streamlit applied theme
    over_theme = {
        "txc_inactive": "black",
        "menu_background": "#b8d7ed",
        "txc_active": "black",
        "option_active": "white",
    }
    font_fmt = {"font-class": "h2", "font-size": "100%"}

    # display a horizontal version of the option bar
    op = hc.option_bar(
        option_definition=option_data,
        key="PrimaryOption",
        override_theme=over_theme,
        font_styling=font_fmt,
        horizontal_orientation=True,
    )
    n_samples_simulation = 1000 

    if op == "Simulation Results":
        n_samples_simulation = st.slider(
                "Number of Simulation Samples:",
                min_value=500,
                max_value=5000,
                value=1000,
                step=100,
            )

    model.samples_rent_increase = np.random.choice(rent_increase_list, n_samples_simulation)
    model.samples_property_price_growth_annual = np.random.choice(property_price_growth_annual_list, n_samples_simulation)
    model.samples_mortgage_interest_annual = np.random.choice(mortgage_interest_annual_list, n_samples_simulation)
    model.samples_investment_return_annual = np.random.choice(investment_return_annual_list, n_samples_simulation)
    model.samples_years_until_sell = np.random.choice(years_until_sell_list, n_samples_simulation)

    model.run_calculations(adjust_for_inflation_bool = adjust_for_inflation_bool)
    #save simulation results
    buying_npv_list = model.buying_npv
    buying_fv_list = model.buying_fv
    renting_fv_list = model.renting_fv
    mortgage_interest_annual_list_chosen = model.samples_mortgage_interest_annual
    property_price_growth_annual_list_chosen = model.samples_property_price_growth_annual
    rent_increase_list_chosen = model.samples_rent_increase
    investment_return_annual_list_chosen = model.samples_investment_return_annual
    years_until_sell_list_chosen = model.samples_years_until_sell

    # typical scenario
    model.samples_rent_increase = np.median(rent_increase_list)
    model.samples_property_price_growth_annual = np.median(property_price_growth_annual_list)
    model.samples_mortgage_interest_annual = np.median(mortgage_interest_annual_list)
    model.samples_investment_return_annual = np.median(investment_return_annual_list)
    model.samples_years_until_sell = int(np.median(years_until_sell_list))
    model.run_calculations(adjust_for_inflation_bool=adjust_for_inflation_bool)

    if model.buying_fv > model.renting_fv:
        text = "Return is typically higher if you <strong>buy</strong>."
        if np.std(buying_fv_list) > np.std(renting_fv_list):
            text += " However, it is less risky if you <strong>rent</strong>."
        else:
            text += " It is also less risky if you <strong>buy</strong>."
    else:
        text = "Return is typically higher if you <strong>rent and invest the deposit</strong>."
        if np.std(buying_fv_list) > np.std(renting_fv_list):
            text += " It is also less risky if you <strong>rent</strong>."
        else:
            text += " However, it is less risky if you <strong>buy</strong>."
    
    sticky_bottom_bar(text)


    

    if op == "Typical Outcome":
        left_column, right_column = st.columns(2)
        with left_column:
            st.write(
                f"### Buy - Asset future value after {model.samples_years_until_sell} years"
            )
            st.markdown(
                f"**Typical Total Asset Value: £{model.buying_fv:,.0f}**",
                help="All components are converted to future value at the time of sale.",
            )
            st.markdown(f"***Breakdown:***")
            st.markdown(f" - Capital Invested (deposit): £{model.DEPOSIT:,.0f}")
            st.markdown(
                f" - Capital Invested (buying cost + stamp duty, if any): £{model.BUYING_COST_FLAT + model.STAMP_DUTY:,.0f}"
            )
            st.markdown(
                f" - Property Price at Sale: :green[£{model.future_house_price:,.0f}]",
                help="Calculated using the property price growth rate set in the left sidebar.",
            )
            st.markdown(
                f" - Selling cost (including Capital Gains Tax): :red[ -£{model.SELLING_COST:,.0f}]",
                help="Total expenses incurred when selling a property. These costs typically include real estate agent commissions, legal fees, advertising expenses, and any necessary repairs or renovations to prepare the property for sale.",
            )
            st.markdown(
                f" - Total maintenance and service costs: :red[ -£{model.fv_ongoing_cost:,.0f}]",
                help="Future value at the time of sale for the total cost associated with maintaining and servicing a property, including expenses such as property management fees, maintenance fees, and other related charges. Assumed to grow at inflation rate. Future value is determined by the discount rate, which is assumed to be equal to the investment return.",
            )
            if model.COUNTRY == "US":
                st.markdown(
                    f" - Total property tax: :red[ -£{model.fv_property_tax:,.0f}]",
                    help="Future value at the time of sale for the total property tax paid",
                )
            st.markdown(
                f" - Total Mortgage Payments: :red[ -£{model.fv_mortgage_payments:,.0f}]",
                help="This is higher than the sum of all mortgage payments since the payments are converted to their future value at the time of sale. Future value is determined by the discount rate, which is assumed to be equal to the investment return.",
            )
            st.markdown(
                f" - Total Rent Saved (future value at time of sale): :green[£{model.rent_fv:,.0f}]",
                help="This is higher than the sum of all rent payments that would have been paid since the payments are converted to their future value at the time of sale. Future value is determined by the discount rate, which is assumed to be equal to the investment return.",
            )

        with right_column:
            st.write(
                f"### Rent and invest - Asset future value after {model.samples_years_until_sell} years"
            )
            st.markdown(
                f"**Typical Total Asset Value: £{model.renting_fv:,.0f}**",
                help="All components are converted to future value at the time of sale.",
            )
            st.markdown(f"***Breakdown:***")
            st.markdown(f" - Capital Invested (deposit): £{model.DEPOSIT:,.0f}")
            st.markdown(
                f" - Capital Invested (buying cost + stamp duty, if any): £{model.BUYING_COST_FLAT + model.STAMP_DUTY:,.0f}"
            )
            st.markdown(
                f" - Capital Gains Tax: :red[-£{model.cgt_investment:,.0f}]",
                help="Your tax rate is determined by the annual salary set in the left sidebar.",
            )
            if (
                model.renting_fv
                - (model.DEPOSIT + model.BUYING_COST_FLAT + model.STAMP_DUTY)
                >= 0
            ):
                st.markdown(
                    f" - Assumed Typical Capital Growth: :green[£{model.renting_fv - (model.DEPOSIT + model.BUYING_COST_FLAT + model.STAMP_DUTY):,.0f}]",
                    help="Calculated with the investment return rated provided in the left sidebar.",
                )
            else:
                st.markdown(
                    f" - Assumed Typical Capital Growth: :red[£{model.renting_fv - (model.DEPOSIT + model.BUYING_COST_FLAT + model.STAMP_DUTY):,.0f}]"
                )

    if op == "Simulation Results":


        plot_hist_from_list(
            [buying_fv_list, renting_fv_list],
            st,
            figsize=(7, 2),
            legends=["Buying", "Renting"],
            main_colors=["orange", "blue"],
            title="Future Asset Value - Simulation Results",
            xlabel="Asset Value",
        )
        st.markdown(
            "<span style='font-size: 14px; font-style: italic;'>Simulation results for future asset value. Using future value at 'years until sell mean' in your assumptions.</span>",
            unsafe_allow_html=True,
        )
        plot_hist_from_list(
            [buying_npv_list],
            st,
            plot_below_zero=True,
            clip=(0, None),
            main_colors=["blue"],
            secondary_color="orange",
            title="Net Present Value of Buying - Simulation Results",
            xlabel="Net Present Value of Buying",
        )
        st.markdown(
            "<span style='font-size: 14px; font-style: italic;'>Negative = Renting is better; Positive = Buying is better.</span>",
            unsafe_allow_html=True,
        )
        st.markdown(
            "<span style='font-size: 14px; font-style: italic;'>Net Present Value represents the net gain/loss that result in purchasing the property in present value. It is calculated as (PV of future house sale price - PV of rent saved - PV of mortgage payments - PV of ongoing costs - deposit - buying costs - stamp duty - PV of selling costs). If it is positive, then it is financially better to buy a property. Present value is calculated using a future discount rate equal to your assumed investment return. This is equivalent to assuming that any amount you save on rent or mortgage will be invested. </span>",
            unsafe_allow_html=True,
        )

        results_dict = {
            "buying_npv": buying_npv_list,
            "mortgage_interest_annual": mortgage_interest_annual_list_chosen,
            "property_price_growth_annual": property_price_growth_annual_list_chosen,
            "rent_increase": rent_increase_list_chosen,
            "investment_return_annual": investment_return_annual_list_chosen,
            "years_until_sell": years_until_sell_list_chosen,
        }
        results_df = pd.DataFrame(results_dict)
        percentiles_df = calculate_percentiles(buying_npv_list, model.DEPOSIT)
        with st.expander("### Net Present Value Statistics", expanded=False):
            st.write(
                f'- Buying is better {100-percentiles_df.loc[5,"Percentile"]:.0f}% of the time'
            )
            st.write(f"- Mean: £{np.mean(buying_npv_list):,.0f}")
            st.write(
                f"- Mean (as % of deposit): {np.mean(buying_npv_list)/model.DEPOSIT*100:.0f}%"
            )
            st.write(f"- Standard Deviation: £{np.std(buying_npv_list):,.0f}")
            st.write(
                f"- Standard Deviation (as % of deposit): {np.std(buying_npv_list)/model.DEPOSIT*100:.0f}%"
            )
            st.write(f"- Skew: {skew(buying_npv_list):.2f}")
        with st.expander(
            "Correlations Between Parameters and Buying NPV", expanded=False
        ):
            st.write(results_df.corr().iloc[0, 1:])
    # return percentiles_df, results_df
