import pandas as pd
import numpy as np
import numpy_financial as npf
from scipy.stats import norm, skew
from utils.general import (
    calculate_percentiles,
    bin_continuous_features,
    get_param_distribution,
)
from utils.finance import (
    get_stamp_duty_next_home,
    annuity_pv,
    annuity_fv,
    annuity_payment,
    pv_future_payment,
    fv_present_payment,
)
import streamlit as st


class Buy_or_Rent_Model:
    def __init__(self) -> None:
        # PARAMS
        # Fixed (in expectation)
        self.COUNTRY = "UK"  # options: UK, US
        self.HOUSE_PRICE = 800000  # including upfront repairs and renovations
        self.RENTAL_YIELD = 0.043  # assumed rent as a proportion of house price https://www.home.co.uk/company/press/rental_yield_heat_map_london_postcodes.pdf
        self.DEPOSIT_MULT = 0.5
        self.MORTGAGE_LENGTH = 30
        self.BUYING_COST_FLAT = (
            3000  # https://www.movingcostscalculator.co.uk/calculator/
        )
        self.SELLING_COST_MULT = (
            0.02  # https://www.movingcostscalculator.co.uk/calculator/
        )
        self.ONGOING_COST_MULT = 0.006  # service charge + repairs, council tax and bills are omitted since they are the same whether buying or renting
        self.ANNUAL_SALARY = 55000
        self.CGT_ALLOWANCE = 6000
        self.PERSONAL_ALLOWANCE = 12570
        # UK Taxes
        self.CGT_BOL = False
        self.CGT_INVESTMENT_BOL = False
        self.STAMP_DUTY_BOL = False
        # US Taxes
        self.PROPERTY_TAX_RATE = 0
        self.MARGINAL_TAX_RATE = 0
        self.US_TAX_RETURN = "Joint Return"  # ['Joint Return','Individual Return']
        # Probability distribution
        self.samples_rent_increase = 0.01325  # historical: https://www.ons.gov.uk/economy/inflationandpriceindices/bulletins/indexofprivatehousingrentalprices/april2023
        self.samples_property_price_growth_annual = 0.025  # historical average = 0.034 over the last 8 year, adjusted down due to end to abnormally low interest rates; source for historical data: https://www.statista.com/statistics/620414/monthly-house-price-index-in-london-england-uk/
        self.samples_mortgage_interest_annual = 0.05
        self.samples_investment_return_annual = 0.06
        self.samples_years_until_sell = 20
        # financial modelling params
        self.inflation = 0.02  # on ongoing costs, also for converting fv
        self.adjust_for_inflation = 0

    def get_capital_gains_tax_property(self):
        cgt = 0
        if self.CGT_BOL:
            taxable_gains = self.future_house_price - self.HOUSE_PRICE
            if self.ANNUAL_SALARY > 50271:
                cgt = taxable_gains * 0.28
            else:
                taxable_income = self.ANNUAL_SALARY - self.PERSONAL_ALLOWANCE
                if taxable_gains - self.CGT_ALLOWANCE + taxable_income <= 50270:
                    cgt = (taxable_gains - self.CGT_ALLOWANCE) * 0.18
                else:
                    cgt += (50271 - taxable_income) * 0.18
                    cgt += (taxable_gains - self.CGT_ALLOWANCE - 50271) * 0.28
        return cgt

    def get_capital_gains_tax_investment(self):
        cgt = 0
        if self.CGT_INVESTMENT_BOL:
            taxable_gains = self.total_investment_fv - self.total_investment
            if self.ANNUAL_SALARY > 50271:
                cgt = taxable_gains * 0.2
            else:
                taxable_income = self.ANNUAL_SALARY - self.PERSONAL_ALLOWANCE
                if taxable_gains - self.CGT_ALLOWANCE + taxable_income <= 50270:
                    cgt = (taxable_gains - self.CGT_ALLOWANCE) * 0.1
                else:
                    cgt += (50271 - taxable_income) * 0.1
                    cgt += (taxable_gains - self.CGT_ALLOWANCE - 50271) * 0.2
        return cgt

    def run_calculations(self, adjust_for_inflation_bool=False):
        self.future_house_price = (
            self.HOUSE_PRICE
            * (1 + self.samples_property_price_growth_annual) ** self.samples_years_until_sell
        )
        self.CGT = self.get_capital_gains_tax_property()
        self.SELLING_COST = self.future_house_price * self.SELLING_COST_MULT + self.CGT
        if self.COUNTRY == "US":
            self.EFFECTIVE_PROPERTY_TAX_RATE = self.PROPERTY_TAX_RATE * (
                1 - self.MARGINAL_TAX_RATE
            )
        if adjust_for_inflation_bool:
            self.adjust_for_inflation = self.inflation
        self.monthly_rent = self.HOUSE_PRICE * self.RENTAL_YIELD / 12
        if self.STAMP_DUTY_BOL:
            self.STAMP_DUTY = get_stamp_duty_next_home(self.HOUSE_PRICE)
        else:
            self.STAMP_DUTY = 0
        self.discount_rate = self.samples_investment_return_annual
        self.DEPOSIT = self.HOUSE_PRICE * self.DEPOSIT_MULT
        self.mortgage_calculations()
        self.get_house_buying_npv()
        self.get_house_buying_fv()
        self.get_renting_fv()

    def mortgage_calculations(self):
        self.mortgage_amount = self.HOUSE_PRICE * (1 - self.DEPOSIT_MULT)
        # assuming 0 growth on mortgage payments
        self.annual_mortgage_payment = annuity_payment(
            self.mortgage_amount, self.samples_mortgage_interest_annual, self.MORTGAGE_LENGTH, 0
        )
        self.pv_mortage_payments = annuity_pv(
            self.annual_mortgage_payment, self.discount_rate, self.MORTGAGE_LENGTH, 0
        )
        self.fv_mortgage_payments = pv_future_payment(
            annuity_fv(
                self.annual_mortgage_payment,
                self.discount_rate,
                self.MORTGAGE_LENGTH,
                0,
            ),
            self.discount_rate,
            self.MORTGAGE_LENGTH - self.samples_years_until_sell,
        ) / float(1 + self.adjust_for_inflation) ** (
            self.samples_years_until_sell
        )  # annuity_fv(self.annual_mortgage_payment, self.discount_rate, self.MORTGAGE_LENGTH, 0)

    def get_house_buying_npv(self):
        self.pv_of_future_house_price = pv_future_payment(
            self.future_house_price, self.discount_rate, self.samples_years_until_sell
        )
        self.pv_of_selling_cost = pv_future_payment(
            self.SELLING_COST, self.discount_rate, self.samples_years_until_sell
        )
        self.pv_ongoing_cost = annuity_pv(
            self.HOUSE_PRICE * self.ONGOING_COST_MULT,
            self.discount_rate,
            self.samples_years_until_sell,
            self.inflation,
        )
        # rent saved
        self.pv_rent_saved = annuity_pv(
            self.HOUSE_PRICE * self.RENTAL_YIELD,
            self.discount_rate,
            self.samples_years_until_sell,
            self.samples_rent_increase,
        )
        # US specific
        if self.COUNTRY == "US":
            self.pv_property_tax = annuity_pv(
                self.HOUSE_PRICE * self.EFFECTIVE_PROPERTY_TAX_RATE,
                self.discount_rate,
                self.samples_years_until_sell,
                self.samples_property_price_growth_annual,
            )
        else:
            self.pv_property_tax = 0
        # sum it up
        self.buying_npv = (
            self.pv_of_future_house_price
            + self.pv_rent_saved
            - self.pv_mortage_payments
            - self.pv_ongoing_cost
            - self.DEPOSIT
            - self.BUYING_COST_FLAT
            - self.STAMP_DUTY
            - self.pv_of_selling_cost
            - self.pv_property_tax
        )

    def get_house_buying_fv(
        self,
    ):  # not accounting for deposit,  immediate costs, and rent saved. ongoing costs and mortgage are rolled up and deducted from fv
        if self.adjust_for_inflation > 0:
            self.SELLING_COST = self.SELLING_COST / float(
                1 + self.adjust_for_inflation
            ) ** (self.samples_years_until_sell)
            self.future_house_price = self.future_house_price / float(
                1 + self.adjust_for_inflation
            ) ** (self.samples_years_until_sell)
        self.fv_ongoing_cost = annuity_fv(
            self.HOUSE_PRICE * self.ONGOING_COST_MULT,
            self.discount_rate,
            self.samples_years_until_sell,
            self.inflation,
            adjust_for_inflation=self.adjust_for_inflation,
        )
        self.rent_fv = annuity_fv(
            self.HOUSE_PRICE * self.RENTAL_YIELD,
            self.discount_rate,
            self.samples_years_until_sell,
            self.samples_rent_increase,
            adjust_for_inflation=self.adjust_for_inflation,
        )
        if self.COUNTRY == "US":
            self.fv_property_tax = annuity_fv(
                self.HOUSE_PRICE * self.EFFECTIVE_PROPERTY_TAX_RATE,
                self.discount_rate,
                self.samples_years_until_sell,
                self.samples_property_price_growth_annual,
            )
        else:
            self.fv_property_tax = 0
        self.buying_fv = (
            self.future_house_price
            + self.rent_fv
            - self.SELLING_COST
            - self.fv_ongoing_cost
            - self.fv_mortgage_payments
            - self.fv_property_tax
        )

    def get_renting_fv(
        self,
    ):  # assumes that buying costs and stamp duty are invested, rent is rolled up and deducted
        self.total_investment = self.BUYING_COST_FLAT + self.STAMP_DUTY + self.DEPOSIT
        self.total_investment_fv = fv_present_payment(
            self.total_investment,
            self.discount_rate,
            self.samples_years_until_sell,
            adjust_for_inflation=self.adjust_for_inflation,
        )
        self.cgt_investment = self.get_capital_gains_tax_investment()
        self.renting_fv = self.total_investment_fv - self.cgt_investment
