import streamlit as st
import numpy as np
from mainbody import generate_main_body
from buy_or_rent import Buy_or_Rent_Model
from utils.general import get_param_distribution
import os
import streamlit_analytics
from config import button_css, n_samples, n_bins

np.random.seed(123)


class App:
    def main_header(self):
        # Title
        st.markdown(
            "<div style=\"font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Helvetica', 'Arial', sans-serif, 'Apple Color Emoji', 'Segoe UI Emoji', 'Segoe UI Symbol'; font-size: 48px; font-weight: light; text-align: center; padding-bottom: 48px;\">Open-source Buy or Rent Calculator</div>",
            unsafe_allow_html=True,
        )
        with st.expander("**About**"):
            st.write(
                "Not sure whether it is financially better to buy a property or rent and invest? We use simulations to show possible returns for buying a property or renting given your assumptions. All parameters are assumed to be uncorrelated."
            )
            st.markdown(
                "***This app is currently in Beta. It assumes that you are in England, UK. Different countries may have different tax rules. Source code: https://huggingface.co/spaces/Edisonymy/buy-or-rent/tree/main***"
            )
        self.country = st.selectbox(
            label="Country", options=["UK", "US"], help="Determines Tax Calculations"
        )
        if self.country == 'US':
            st.markdown(
                "***The US version of this app is currently in Beta. US taxes are not fully implemented and the defaults values do not currently reflect US norms.***"
            )

    def get_sidebar(self):
        # User-enterable parameters for data generation
        st.sidebar.subheader("Your Assumptions")
        self.house_price = st.sidebar.number_input(
            "Property Price", value=300000, step=5000
        )
        self.monthly_rent = st.sidebar.number_input(
            "Monthly Rent:", value=int(self.house_price * 0.043 / 12), step=100
        )
        self.rental_yield = (12 * self.monthly_rent) / self.house_price
        st.sidebar.markdown(
            f"<span style='font-size: 12px;'>Implied Rental Yield: {self.rental_yield*100:.2f}%</span>",
            unsafe_allow_html=True,
        )
        text = "Checkout a rental yield map here: https://www.home.co.uk/company/press/rental_yield_heat_map_london_postcodes.pdf"
        st.sidebar.markdown(
            f"<span style='font-size: 11px;'>{text}</span>", unsafe_allow_html=True
        )
        self.deposit = st.sidebar.slider(
            "Deposit:",
            min_value=0,
            max_value=self.house_price,
            value=int(0.3 * self.house_price),
            step=100,
        )
        self.deposit_mult = self.deposit / self.house_price
        st.sidebar.subheader("Advanced Model Parameters:")
        st.sidebar.markdown(
            "<span style='font-size: small; color: gray;'>It's hard to predict the future, so this section allows the simulations to reflect your uncertainty. The more uncertain you are about a parameter, the higher the standard deviation (sd) you should assume.</span>",
            unsafe_allow_html=True,
        )
        self.mortgage_interest_annual_mean = st.sidebar.slider(
            "Mortgage Interest Rate Mean:",
            min_value=0.0,
            max_value=10.0,
            value=5.5,
            step=0.1,
            format="%f%%",
        )
        self.mortgage_interest_annual_std = st.sidebar.slider(
            "Mortgage Interest Rate sd:",
            min_value=0.0,
            max_value=10.0,
            value=1.2,
            step=0.1,
            format="%f%%",
        )
        text = "Check out historical mortgage rates here: https://tradingeconomics.com/united-kingdom/mortgage-rate"
        st.sidebar.markdown(
            f"<span style='font-size: 11px;'>{text}</span>", unsafe_allow_html=True
        )
        self.mortgage_interest_annual_list = (
            get_param_distribution(
                self.mortgage_interest_annual_mean,
                self.mortgage_interest_annual_std,
                n_samples,
                sidebar=True,
                xlabel="Average Mortgage Interest Rate",
            )
            / 100
        )
        st.sidebar.write("---")
        self.investment_return_annual_mean = st.sidebar.slider(
            "Investment Return Mean:",
            min_value=0.0,
            max_value=20.0,
            value=6.0,
            step=0.1,
            format="%f%%",
        )
        self.investment_return_annual_std = st.sidebar.slider(
            "Investment Return sd:",
            min_value=0.0,
            max_value=5.0,
            value=02.0,
            step=0.1,
            format="%f%%",
        )
        text = "Check out historical stock market returns here: https://www.investopedia.com/ask/answers/042415/what-average-annual-return-sp-500.asp"
        st.sidebar.markdown(
            f"<span style='font-size: 11px;'>{text}</span>", unsafe_allow_html=True
        )
        self.investment_return_annual_list = (
            get_param_distribution(
                self.investment_return_annual_mean,
                self.investment_return_annual_std,
                n_samples,
                sidebar=True,
                xlabel="Average Investment Rate of Return",
            )
            / 100
        )
        with st.sidebar.expander("**Additional Parameters:**"):
            if self.country == "UK":
                self.stamp_duty_bol = st.checkbox(
                    "I pay Stamp Duty.",
                    value=False,
                    help="Stamp Duty (UK): A tax paid by the buyer when purchasing property in the United Kingdom. The amount of Stamp Duty depends on the property's purchase price and may include additional rates for second homes and buy-to-let properties.",
                )
                self.cgt_bol = st.checkbox(
                    "I pay capital gains tax on the property.",
                    value=False,
                    help="Capital Gains Tax (CGT): A tax levied on the profit (capital gain) made from the sale or disposal of assets such as property, investments, or valuable possessions. The amount of CGT owed is typically calculated based on the difference between the purchase price and the selling price of the asset. Different rates may apply to individuals and businesses, and there are often exemptions and allowances that can reduce the taxable amount.",
                )
                self.cgt_investment_bol = st.checkbox(
                    "I pay capital gains tax on the investment.",
                    value=False,
                    help="Capital Gains Tax (CGT): A tax levied on the profit (capital gain) made from the sale or disposal of assets such as property, investments, or valuable possessions. The amount of CGT owed is typically calculated based on the difference between the purchase price and the selling price of the asset. Different rates may apply to individuals and businesses, and there are often exemptions and allowances that can reduce the taxable amount.",
                )
                self.annual_income = st.number_input(
                    "Annual Salary",
                    value=20000,
                    step=100,
                    help="This should be set to the expected salary at time of sale, required to calculate capital gains tax.",
                )
            if self.country == "US":
                self.us_tax_return = st.selectbox(
                    label="How you file your taxes",
                    options=["Joint Return", "Individual Return"],
                    help="The initial $250,000 in profit gained from selling your home is exempt from taxation. If you file jointly with your spouse, this exemption increases to $500,000. Additional details can be found in IRS Topic 701.",
                )
                self.property_tax_rate = st.slider(
                    "Property Tax Rate:",
                    min_value=0.0,
                    max_value=10.0,
                    value=1.35,
                    step=0.01,
                    format="%f%%",
                )
                st.markdown(
                    f"<span style='font-size: small; color: gray;'>${self.property_tax_rate/self.house_price/100:.2f}% for first year</span>",
                    unsafe_allow_html=True,
                )
                self.marginal_tax_rate = st.slider(
                    "Marginal Tax Rate:",
                    min_value=0,
                    max_value=60,
                    value=20,
                    step=1,
                    format="%f%%",
                )
            self.ongoing_cost = st.number_input(
                "Annual combined maintenance, insurance and service charge:",
                value=int(self.house_price * 0.006),
                step=100,
                help="The total annual cost associated with maintaining, insurance, and servicing a property, including expenses such as property management fees, maintenance fees, common fees, and other similar charges. Assumed to grow at rate of inflation.",
            )
            st.markdown(
                f"<span style='font-size: small; color: gray;'>{self.ongoing_cost/self.house_price*100:.2f}% of Property Price</span>",
                unsafe_allow_html=True,
            )
            self.buying_cost = st.number_input(
                "Buying cost (excluding taxes):",
                value=3000,
                step=100,
                help="This includes laywer's fee, mortgage lender fees, surveyor's fee, valuation fees and other fees you expect to pay at the time of purchase.",
            )
            st.markdown(
                f"<span style='font-size: small; color: gray;'>{self.buying_cost/self.house_price*100:.2f}% of Property Price</span>",
                unsafe_allow_html=True,
            )
            self.selling_cost_no_cgt = (
                st.number_input(
                    "Selling cost as a % of property value (excluding Taxes):",
                    value=2.0,
                    step=0.1,
                    format="%.1f",
                    help="Total expense incurred when selling a property. This typically include real estate agent commissions, legal fees, advertising expenses, and any necessary repairs or renovations to prepare the property for sale.",
                )
                / 100
            )
            st.markdown(
                f"<span style='font-size: small; color: gray;'>{self.selling_cost_no_cgt*self.house_price:.0f}</span>",
                unsafe_allow_html=True,
            )
            self.inflation = (
                st.number_input(
                    "Inflation:",
                    min_value=0.0,
                    max_value=100.0,
                    value=2.0,
                    step=0.1,
                    format="%.1f",
                    help="Used for inflation adjustment only. It does not affect the model.",
                )
                / 100
            )
            self.mortgage_length = st.slider(
                "Mortgage Length:", min_value=1, max_value=40, value=30, step=1
            )
            # uncertain parameters
            st.write("---")
            self.property_price_growth_annual_mean = st.slider(
                "Property Price Growth Mean:",
                min_value=-5.0,
                max_value=10.0,
                value=3.0,
                step=0.1,
                format="%f%%",
            )
            self.property_price_growth_annual_std = st.slider(
                "Property Price Growth sd:",
                min_value=0.0,
                max_value=5.0,
                value=1.0,
                step=0.1,
                format="%f%%",
            )
            text = "Check out historical property price growth here: https://www.ons.gov.uk/economy/inflationandpriceindices/bulletins/housepriceindex/june2023"
            st.markdown(
                f"<span style='font-size: 11px;'>{text}</span>", unsafe_allow_html=True
            )
            self.property_price_growth_annual_list = (
                get_param_distribution(
                    self.property_price_growth_annual_mean,
                    self.property_price_growth_annual_std,
                    n_samples,
                    xlabel="Annual Property Value Growth",
                )
                / 100
            )
            st.write("---")
            self.rent_increase_mean = st.slider(
                "Rent Increase Mean:",
                min_value=0.0,
                max_value=10.0,
                value=2.0,
                step=0.1,
                format="%f%%",
            )
            self.rent_increase_std = st.slider(
                "Rent Increase sd:",
                min_value=0.0,
                max_value=5.0,
                value=1.0,
                step=0.1,
                format="%f%%",
            )
            text = "Checkout historical rent increases here: https://www.ons.gov.uk/economy/inflationandpriceindices/bulletins/indexofprivatehousingrentalprices/july2023"
            st.markdown(
                f"<span style='font-size: 11px;'>{text}</span>", unsafe_allow_html=True
            )
            self.rent_increase_list = (
                get_param_distribution(
                    self.rent_increase_mean,
                    self.rent_increase_std,
                    n_samples,
                    xlabel="Average Annual Rent Increase",
                )
                / 100
            )
            st.write("---")
            self.years_until_sell_mean = st.slider(
                "Years Until Sell Mean:", min_value=0, max_value=100, value=15
            )
            self.years_until_sell_std = st.slider(
                "Years Until Sell sd:", min_value=0, max_value=10, value=5
            )
            self.years_until_sell_list = get_param_distribution(
                self.years_until_sell_mean,
                self.years_until_sell_std,
                n_samples,
                as_int=True,
                xlabel="Years Until Property Is Sold",
            )
            st.write("---")

    def support_section(self):
        st.write(
            "If you found this useful and want to support me, consider buying me a coffee here:"
        )
        # Embed the "Buy Me a Coffee" button script with custom CSS
        buy_me_a_coffee_html = """
        <script type="text/javascript" src="https://cdnjs.buymeacoffee.com/1.0.0/button.prod.min.js"
            data-name="bmc-button" data-slug="edisonymyt" data-color="#f0ead8" data-emoji="☕" data-font="Cookie"
            data-text="Buy me a coffee" data-outline-color="#000000" data-font-color="#000000"
            data-coffee-color="#ffffff" ></script>
        """
        # Display the custom CSS and the button in the Streamlit app
        st.components.v1.html(buy_me_a_coffee_html)

    def main(self):
        with streamlit_analytics.track(unsafe_password=os.getenv("sa_password")):
            if "sidebar_state" not in st.session_state:
                st.session_state.sidebar_state = "auto"
            st.set_page_config(
                page_title="Buy or Rent Calculator",
                initial_sidebar_state=st.session_state.sidebar_state,
            )
            # Apply the custom CSS
            st.markdown(button_css, unsafe_allow_html=True)

            self.main_header()
            self.get_sidebar()

            # Initialize the model
            model = Buy_or_Rent_Model()
            model.COUNTRY = self.country
            model.HOUSE_PRICE = self.house_price
            model.DEPOSIT_MULT = self.deposit_mult
            model.RENTAL_YIELD = self.rental_yield
            model.SELLING_COST_MULT = self.selling_cost_no_cgt
            model.MORTGAGE_LENGTH = self.mortgage_length
            model.ONGOING_COST_MULT = self.ongoing_cost / self.house_price
            model.BUYING_COST_FLAT = self.buying_cost
            model.inflation = self.inflation
            if self.country == "UK":
                model.ANNUAL_SALARY = self.annual_income
                model.STAMP_DUTY_BOL = self.stamp_duty_bol
                model.CGT_BOL = self.cgt_bol
                model.CGT_INVESTMENT_BOL = self.cgt_investment_bol
            if self.country == "US":
                model.US_TAX_RETURN = self.us_tax_return
                model.PROPERTY_TAX_RATE = self.property_tax_rate / 100
                model.MARGINAL_TAX_RATE = self.marginal_tax_rate / 100

            # Generate combinations and calculate NPV
            generate_main_body(
                model,
                mortgage_interest_annual_list=self.mortgage_interest_annual_list,
                property_price_growth_annual_list=self.property_price_growth_annual_list,
                rent_increase_list=self.rent_increase_list,
                investment_return_annual_list=self.investment_return_annual_list,
                years_until_sell_list=self.years_until_sell_list,
            )
            # st.markdown("***To adjust your assumptions, please use the left sidebar (mobile users please press the top left button).***")
            if st.button("Adjust Assumptions (Toggle Sidebar)"):
                st.session_state.sidebar_state = (
                    "collapsed"
                    if st.session_state.sidebar_state == "expanded"
                    else "expanded"
                )
                # Force an app rerun after switching the sidebar state.
                st.experimental_rerun()

            st.write("---")
            self.support_section()


if __name__ == "__main__":
    app = App()
    app.main()
