from torch.utils.data import DataLoader
from deepest.modules import Parameter2dNet
from deepest.datasets import InferenceDelayDataset
from deepest.metrics import match_components
import numpy as np

class Runner:
    def __init__(self, model: str, dataset: str, bs: int, num_worker: int):
        self.module = Parameter2dNet.from_file(f"{model}")
        self.dataset_config = self.module.get_datasetconfig()
        self.dataset = InferenceDelayDataset(path=dataset, **self.dataset_config)
        self.bs = bs
        self.num_worker = num_worker
                
    def _preallocate(self, data_shape: tuple[int, ...], eta_shape: tuple[int, ...]):
        data = np.empty((len(self), *data_shape), dtype=np.complex128)
        truth = np.empty((len(self), *eta_shape))
        estim = np.empty((len(self), *eta_shape))
        return data, truth, estim
    
    def _get_batchrange_for_index(self, ii: int):
        start_idx = ii*self.bs
        stop_idx = (ii+1)*self.bs
        if stop_idx > len(self.dataset):
            stop_idx = len(self.dataset)
                
        return range(start_idx, stop_idx)
            
    def run(self, snr: int) -> tuple[np.ndarray, np.ndarray, np.ndarray]:
        self.dataset.noise_var = (snr, snr)
        dataloader = DataLoader(
            self.dataset,
            batch_size=self.bs,
            num_workers=self.num_worker,
            worker_init_fn=lambda worker_id: np.random.seed(worker_id),
            shuffle=False,
        )

        for ii, (x, _, z) in enumerate(dataloader):
            z = z[0][:, :2, :]
            if ii == 0:
                data, truth, estim = self._preallocate(x.shape[1:], z.shape[1:])
                
            idx_range = self._get_batchrange_for_index(ii)
            
            data[idx_range] = x.cpu().numpy()
            truth[idx_range] = z.cpu().numpy()
            estim[idx_range] = self.module.fit(x)[:, :2, :]
        
        estim, truth = match_components(estim, truth) 
        
        return data, truth, estim
    
    def fit(self, data: np.ndarray) -> np.ndarray:
        x = self.module.fit(data)
        return x[:, :2, :]
    
    def __len__(self):
        return len(self.dataset)