# https://huggingface.co/St0nedB/deepest-public
import os
import sys
import subprocess
import toml
from argparse import Namespace
import numpy as np
import logging
import gradio as gr
import matplotlib
import matplotlib.pyplot as plt
from huggingface_hub import hf_hub_download

matplotlib.use("Agg")
logger = logging.basicConfig(level=logging.ERROR)

# define global variable demos
DATA_SHAPE = (64,64)
ETA_SHAPE = (2, 20)
DATASET = "./data"
N = 1000
BS = 256
WORKER = 2
SNRS = {
    "0": 1.0,
    "10": 0.1,
    "20": 0.01,
    "30": 0.001,
}

# download model from huggingface hub
MODEL_PATH = hf_hub_download("St0nedB/deepest-demo", "2022.07.03.2338.param2d.model", use_auth_token=os.environ["MODEL_TOKEN"])
RUNNER = None

# preallocated result arrays
DATA = np.empty((len(SNRS), N, *DATA_SHAPE), dtype=np.complex128)
TRUTH = np.empty((len(SNRS), N, *ETA_SHAPE))
ESTIM = np.empty((len(SNRS), N, *ETA_SHAPE))

# load texts
TEXTS = Namespace(**toml.load("texts.toml"))

def install_deepest():
    git_token = os.environ["GIT_TOKEN"]
    git_url = os.environ["GIT_URL"]
    git_commit = os.environ["GIT_COMMIT"]
    subprocess.check_call([sys.executable, "-m", "pip", "install", f"git+https://hggn:{git_token}@{git_url}@{git_commit}"])
    return


def make_plots(snr: float, idx: int):
    idx -= 1
    data, truth, estim = DATA[snr][idx], TRUTH[snr][idx], ESTIM[snr][idx]
    
    fig_data = make_dataplot(data)
    fig_param = make_parameterplot(estim, truth)
    
    return fig_data, fig_param

def make_dataplot(x: np.ndarray):
    plt.close()
    x = np.rot90(10*np.log10(np.abs(np.fft.fftn(x))), k=-1)
    fig, ax = plt.subplots(1,1)
    ax.imshow(x, extent=[0,1,0,1], cmap="viridis")
    ax.set_xlabel("Norm. Delay")
    ax.set_ylabel("Norm. Doppler")
    
    return fig

def make_parameterplot(estim: np.ndarray, truth: np.ndarray = None, **kwargs):
    plt.close()
    fig, ax = plt.subplots(1,1)
    ax = plot_parameters(ax, es=estim, gt=truth, **kwargs)
    ax.set_xlim(0,1)
    ax.set_ylim(0,1)
    
    return fig

def load_numpy(file_obj) -> None | np.ndarray:
    if file_obj is None:
        # no file given
        return None
    
    file = file_obj.name
    if not(os.path.splitext(file)[1] in [".npy", ".npz"]):
        # no numpy file
        return None
    
    data = np.load(file)
    if len(data.shape) != 3:
        # not in proper shape
        return None
    
    return data
    
def process_user_input(file_obj):
    data = load_numpy(file_obj)
    if data is None:
        return None
    
    return gr.update(minimum=1, step=1, maximum=len(data), visible=True, value=1)

def make_user_plot(file_obj, idx: int):
    idx -= 1
    data = load_numpy(file_obj)
    
    estim = RUNNER.fit(data[idx][None,])
    bg_data = np.rot90(10*np.log10(np.abs(np.fft.fftn(data[idx], norm="ortho"))), k=-1)
    fig_estim = make_parameterplot(estim=estim[0], bg=bg_data, extent=[0,1,0,1], cmap="viridis")
    
    return fig_estim
    

def demo():
    with gr.Blocks() as demo:
        gr.Markdown(
            TEXTS.introduction
        )
        
        with gr.Column():
            snr = gr.Radio(choices=["0", "10", "20", "30"], type="index", value="0", label="SNR [dB]")
            
        with gr.Row():
            data = gr.Plot(label="Data")
            result = gr.Plot(label="Results")
            
        with gr.Row():
            slider = gr.Slider(1, N, 1, label="Sample Index")
            
        # update callbacks
        slider.change(make_plots, [snr, slider], [data, result])
        snr.change(make_plots, [snr, slider], [data, result])
        
        with gr.Column():
            gr.Markdown(
                TEXTS.try_your_own
            )
            
            with gr.Row():
                with gr.Column():
                    user_file = gr.File(file_count="single", type="file", interactive=True)
                    run_btn = gr.Button("Run Inference")
                
                user_plot = gr.Plot(label="Results")
                
            with gr.Column():
                user_slider = gr.Slider(visible=False, label="Sample Index")
                
        run_btn.click(process_user_input, [user_file], [user_slider], show_progress=True)
        user_slider.change(make_user_plot, [user_file, user_slider], [user_plot])
        
        gr.Markdown(
            TEXTS.acknowledgements
        )
        
        gr.Markdown(
            TEXTS.contact
        )
        
    demo.launch()
    
def main():
    for dd, snr in enumerate(SNRS.values()):
        DATA[dd], TRUTH[dd], ESTIM[dd] = RUNNER.run(snr=snr)
        
    demo()
    
    
if __name__ == "__main__":
    try: 
        import deepest
    except ModuleNotFoundError:
        install_deepest()
        
    from deepest.utils import plot_parameters
    from helper import Runner
    RUNNER = Runner(MODEL_PATH, DATASET, BS, WORKER)
    main()
