#References: 1. https://www.kdnuggets.com/2021/03/speech-text-wav2vec.html 
            #2. https://www.youtube.com/watch?v=4CoVcsxZphE 
            #3. https://www.analyticsvidhya.com/blog/2021/02/hugging-face-introduces-the-first-automatic-speech-recognition-model-wav2vec2/ 

#Importing all the necessary packages
import nltk
import librosa
import torch
import gradio as gr
from transformers import Wav2Vec2Tokenizer, Wav2Vec2ForCTC
nltk.download("punkt")

#Loading the model and the tokenizer
model_name = "facebook/wav2vec2-base-960h"
tokenizer = Wav2Vec2Tokenizer.from_pretrained(model_name)
model = Wav2Vec2ForCTC.from_pretrained(model_name)


def load_data(input_file):
  
  """ Function for resampling to ensure that the speech input is sampled at 16KHz.
  """
  #read the file
  speech, sample_rate = librosa.load(input_file)
  #make it 1-D
  if len(speech.shape) > 1: 
      speech = speech[:,0] + speech[:,1]
  #Resampling at 16KHz since wav2vec2-base-960h is pretrained and fine-tuned on speech audio sampled at 16 KHz.
  if sample_rate !=16000:
    speech = librosa.resample(speech, sample_rate,16000)
  return speech
  
  

def correct_casing(input_sentence):
  """ This function is for correcting the casing of the generated transcribed text
  """
  sentences = nltk.sent_tokenize(input_sentence)
  return (' '.join([s.replace(s[0],s[0].capitalize(),1) for s in sentences]))
  


def asr_transcript(input_file):
  """This function generates transcripts for the provided audio input
  """
  speech = load_data(input_file)
  
  #Tokenize
  input_values = tokenizer(speech, return_tensors="pt").input_values
  #Take logits
  logits = model(input_values).logits
  #Take argmax
  predicted_ids = torch.argmax(logits, dim=-1)
  #Get the words from predicted word ids
  transcription = tokenizer.decode(predicted_ids[0])
  #Output is all upper case
  transcription = correct_casing(transcription.lower())
  return transcription
  

gr.Interface(asr_transcript,
             inputs = gr.inputs.Audio(source="microphone", type="filepath", optional=True, label="Please record your voice"),
             outputs = gr.outputs.Textbox(label="Output Text"),
             title="ASR using Wav2Vec 2.0",
             description = "This application displays transcribed text for given audio input",
             examples = [["Test_File1.wav"], ["Test_File2.wav"], ["Test_File3.wav"]], theme="grass").launch()

