#!/bin/bash
set -euo pipefail

function download-model() {
    local _option=$1
    local _filename=$2
    local _url=$3
    local _dir

    ! [ $# -eq 3 ] && (echo "usage: "; for o in checkpoint lora vae control-net embedding; do echo "  \$ download-model --$o <filename> <url>"; done) || true
    [ $# -eq 0 ] && return 0 || ! [ $# -eq 3 ] && (echo ""; echo "error - invalid number of arguments (expected 3, received $#)"; echo -n "\$ download-model $1"; (for arg in "${@: 2}"; do echo -n " \"${arg//\"/\\\"}\""; done) && echo "") && return 1 || true

    case ${_option,,} in
        --checkpoint) _dir="/app/stable-diffusion-webui/models/Stable-diffusion";;
        --lora) _dir="/app/stable-diffusion-webui/models/Lora";;
        --vae) _dir="/app/stable-diffusion-webui/models/VAE";;
        --control-net) _dir="/app/stable-diffusion-webui/models/ControlNet";;
        --embedding) _dir="/app/stable-diffusion-webui/embeddings";;

        *) echo "error - unknown first argument: '$1' (valid options are --checkpoint, --lora, --vae, --control-net or --embedding):"; echo "\$ download-model $1 \"$2\" \"$3\""; return 1;;
    esac

    echo "\$ download-model $_option \"$2\" \"$3\"" ; echo ""
    aria2c --console-log-level=error -c -x 16 -s 16 -k 1M $_url -d $_dir -o $_filename && echo ""
}

## ----------------------------

## Adds a header to the webui on Hugging Face Spaces.
## sed -i -e '/demo:/r /app/stable-diffusion-webui/header_patch.py' /app/stable-diffusion-webui/modules/ui.py

## ----------------------------

## Installing less models if $IS_SHARED_UI environment variable is set.
if [ ${IS_SHARED_UI:-0} != 0 ]; then
    download-model --checkpoint "v1-5-pruned-emaonly.safetensors" "https://huggingface.co/runwayml/stable-diffusion-v1-5/resolve/39593d5650112b4cc580433f6b0435385882d819/v1-5-pruned-emaonly.safetensors"
    download-model --checkpoint "v1-5-pruned-emaonly.yaml" "https://huggingface.co/runwayml/stable-diffusion-v1-5/resolve/39593d5650112b4cc580433f6b0435385882d819/v1-inference.yaml"
    download-model --control-net "cldm_v15.yaml" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/cldm_v15.yaml"
    download-model --control-net "control_canny-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_canny-fp16.safetensors"
    download-model --control-net "control_depth-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_depth-fp16.safetensors"
    download-model --control-net "control_normal-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_normal-fp16.safetensors"
    download-model --control-net "control_openpose-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_openpose-fp16.safetensors"
    download-model --control-net "control_scribble-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_scribble-fp16.safetensors"
    download-model --checkpoint "AtoZovyaRPGArtistTools15_sd15V1.safetensors" "https://civitai.com/api/download/models/10185"
    download-model --embedding "bad_prompt_version2.pt" "https://huggingface.co/datasets/Nerfgun3/bad_prompt/resolve/72fd9d6011c2ba87b5847b7e45e6603917e3cbed/bad_prompt_version2.pt"
    sed -i -e '/(modelmerger_interface, \"Checkpoint Merger\", \"modelmerger\"),/d' /app/stable-diffusion-webui/modules/ui.py
    sed -i -e '/(train_interface, \"Train\", \"ti\"),/d' /app/stable-diffusion-webui/modules/ui.py
    sed -i -e '/extensions_interface, \"Extensions\", \"extensions\"/d' /app/stable-diffusion-webui/modules/ui.py
    sed -i -e '/settings_interface, \"Settings\", \"settings\"/d' /app/stable-diffusion-webui/modules/ui.py
    rm -rf /app/stable-diffusion-webui/scripts /app/stable-diffusion-webui/extensions/deforum-for-automatic1111-webui /app/stable-diffusion-webui/extensions/stable-diffusion-webui-images-browser /app/stable-diffusion-webui/extensions/sd-civitai-browser /app/stable-diffusion-webui/extensions/sd-webui-additional-networks
    cp -f shared-config.json config.json
    cp -f shared-ui-config.json ui-config.json
    exit 0
fi
## End of lightweight installation for $IS_SHARED_UI setup.

## ----------------------------
## env $IS_SHARED_UI is not set
## ----------------------------

## ----------------------------

## LoRA (low-rank adaptation) · epi_noiseoffset v2:

## ----------------------------

## VAE (variational autoencoder):
##  MSE: Smoother images
download-model --vae "vae-ft-mse-840000-ema-pruned.safetensors" "https://huggingface.co/stabilityai/sd-vae-ft-mse-original/resolve/main/vae-ft-mse-840000-ema-pruned.safetensors"
##  EMA: Sharper images
download-model --vae "vae-ft-ema-560000-ema-pruned.safetensors" "https://huggingface.co/stabilityai/sd-vae-ft-ema-original/resolve/main/vae-ft-ema-560000-ema-pruned.safetensors"
##  Unknown
download-model --vae "Grapefruit.vae.pt" "https://huggingface.co/iZELX1/Grapefruit/resolve/main/Grapefruit.vae.pt"
##  Anime
download-model --vae "kl-f8-anime.ckpt" "https://huggingface.co/hakurei/waifu-diffusion-v1-4/resolve/main/vae/kl-f8-anime.ckpt"

## ----------------------------

## ControlNet · Pre-extracted models:
#download-model --control-net "cldm_v15.yaml" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/cldm_v15.yaml"
#download-model --control-net "cldm_v21.yaml" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/cldm_v21.yaml"
#download-model --control-net "control_canny-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_canny-fp16.safetensors"
#download-model --control-net "control_depth-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_depth-fp16.safetensors"
#download-model --control-net "control_hed-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_hed-fp16.safetensors"
#download-model --control-net "control_normal-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_normal-fp16.safetensors"
#download-model --control-net "control_openpose-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_openpose-fp16.safetensors"
#download-model --control-net "control_scribble-fp16.safetensors" "https://huggingface.co/webui/ControlNet-modules-safetensors/resolve/87c3affbcad3baec52ffe39cac3a15a94902aed3/control_scribble-fp16.safetensors"

## ----------------------------

## Embedding
##   Bad-hands-5
download-model --embedding "bad-hands-5.pt" "https://huggingface.co/yesyeahvh/bad-hands-5/resolve/main/bad-hands-5.pt"
##   FastNegative
download-model --embedding "FastNegativeEmbedding.pt" "https://civitai.com/api/download/models/76712"

## ----------------------------

## Checkpoints:
##  Anything
download-model --checkpoint "anything-v3-vae-swapped.ckpt" "https://huggingface.co/ckpt/anything-v3-vae-swapped/resolve/main/anything-v3-vae-swapped.ckpt"

## ----------------------------

## Add additional models that you want to install on startup. Replace URL and FILENAME from the examples below with your values.

## Usage:
## download-model --checkpoint <filename> <url>
## download-model --lora <filename> <url>
## download-model --vae <filename> <url>
## download-model --control-net <filename> <url>
## download-model --embedding <filename> <url>

## ----------------------------


## Checkpoint · Example:
# download-model --checkpoint "FILENAME" "URL"

## LORA (low-rank adaptation) · Example:
# download-model --lora "FILENAME" "URL"

## VAE (variational autoencoder) · Example:
# download-model --vae "FILENAME" "URL"
