import os

import gradio as gr
import wandb

from my_recommending.app import build_app_blocks, MovieMarkdownGenerator
from my_recommending.movielens.lit import MovieLensNonGradientRecommender
from my_recommending.movielens.data import MovieLens25m


project = "Recommending"
tmdb_api_token = os.environ["TMDB_API_TOKEN"]
lightning_class = MovieLensNonGradientRecommender
config = dict(
    model_artifact="my_mf_slim_movielens_25m:v9",
    data_artifact="movielens25m:v4",
    media_directory="media",
    n_recommendations_from_imdb_ratings=50,
)

wandb.init(job_type="app", project=project, config=config)

model_artifact = wandb.use_artifact(config["model_artifact"])
checkpoint_path = model_artifact.file()
lightning_module = lightning_class.load_from_checkpoint(
    checkpoint_path, map_location="cpu"
)
recommender = lightning_module.model.eval()

data_artifact = wandb.use_artifact(config["data_artifact"])
data_directory = data_artifact.download()
movielens = MovieLens25m(directory=data_directory)
movie_markdown_generator = MovieMarkdownGenerator(
    movielens=movielens, tmdb_api_token=tmdb_api_token
)

with gr.Blocks() as app:
    build_app_blocks(
        recommender=recommender,
        movie_markdown_generator=movie_markdown_generator,
        media_directory=config["media_directory"],
        n_recommendations_from_imdb_ratings=config[
            "n_recommendations_from_imdb_ratings"
        ],
    )

app.launch()
