# Title:  arxiv_info.py
# Author: Didier Guillevic
# Date:   2021

#
# Utility code to load information about the arXiv papers.
#
import streamlit as st
import pandas as pd
import h5py
import torch
import sentence_transformers as sbert
import os

import logging   # https://docs.python.org/3/howto/logging.html
logging.basicConfig(format='%(levelname)s:%(message)s', level=logging.INFO)

#
# arxiv paper embeddings AND model used to generate those embeddings
#
class PaperEmbeddings:
    """Embeddings and associated metadata information"""

    def __init__(self, embeddings_fName):
        # INIT
        self.embeddings_fName = None
        self.model_name = None
        self.model_max_seq_length = None
        self.paper_embeddings = None      # Embeddings of each document in the collection
        self.paper_ids = None             # ID of each document (arxiv paper ID)
        self.model = None                 # SentenceTransformer model use to encode the documents

        # LOAD
        logging.info("LOADING the document embeddings")
        self.load_embeddings(embeddings_fName)
        logging.info("LOADING the SentenceTransformer model")
        self.load_model(self.model_name, self.model_max_seq_length)

    def load_embeddings(self, embeddings_fName):

        self.embeddings_fName = embeddings_fName

        # OPEN embeddings file
        hf = h5py.File(embeddings_fName, 'r')

        #print("List of keys: {}".format(hf.keys()))
        self.paper_embeddings     = hf['title_abstract_embeddings']
        self.model_name           = self.paper_embeddings.attrs['model_name']
        self.model_max_seq_length = self.paper_embeddings.attrs['model_max_seq_length']
        self.paper_ids            = hf['paper_ids']

        # COPY the data so we can close the file
        self.paper_embeddings = self.paper_embeddings[()]
        self.paper_ids        = self.paper_ids[()]

        # CLOSE embeddings file
        hf.close()

        # Convert the embeddings to PyTorch tensors
        self.paper_embeddings = torch.from_numpy(self.paper_embeddings)

        # Move to GPU if available
        if torch.cuda.is_available():
            self.paper_embeddings = self.paper_embeddings.to('cuda')

        # Normalize the embeddings so we don't need to normalize each time we use cos_sim.
        # Instead, we can save time and only compute the dot product (use dot_score).
        self.paper_embeddings = sbert.util.normalize_embeddings(self.paper_embeddings)

    def load_model(self, model_name, model_max_seq_length):
        self.model = sbert.SentenceTransformer(model_name)  # SentenceTransformer model
        self.model.max_seq_length = model_max_seq_length

    def info(self):
        info_strings = []

        info_strings.append("\tpaper_embeddings.shape: {}".format(self.paper_embeddings.shape))
        info_strings.append("\tembdding of first paper: {}...".format(self.paper_embeddings[0][:5]))
        info_strings.append("\tmodel_name: {}".format(self.model_name))
        info_strings.append("\tmodel_max_seq_length: {}".format(self.model_max_seq_length))
        info_strings.append("\tpaper_ids.shape: {}".format(self.paper_ids.shape))
        info_strings.append("\tpaper_ids: {}...".format(self.paper_ids[:5]))

        return "\n".join(info_strings)

#
# Load arxiv metadata from a File
#
@st.cache
def load_arxiv_meta_data(fName):
    df = pd.read_json(fName, lines=True)
    return df

#
#
#
def get_paper_title_abstract(df, paper_id):
    """ GET the tile and abstract of a paper given its ID
    Args:
        df (pandas dataframe): contains the metadata information of arxiv papers
        paper_id (str): The arxiv paper ID

    Returns:
        (title, abstract): The title and the abstract for given paper ID. None if not found.
    """
    if df.id.isin([paper_id]).any():
        title = df.loc[df.id == paper_id, 'title'].values[0]
        abstract = df.loc[df.id == paper_id, 'abstract'].values[0]
        return (title, abstract)

    return (None, None)

#
#
#
@st.cache
def find_similar_docs(query, embeddings, nb_results=10):
    """ FIND similar documents to the given query
    Get the sentence embedding for given query.
    Find the "closest" sentences in our collection.

    Args:
        query (str): Query should be as long as possible... (At least a sentence or more...)
        embeddings (PaperEmbeddings): Embeddings of collection and embedding model
        nb_results: number of similar documents (neighbours) to return

    Returns:
        hits: the results for the given query
    """
    queries = [query.strip()] # one single query passed to the function

    # Get the embeddings for all of our queries (a single query in our case)
    queries_embeddings = embeddings.model.encode(queries, convert_to_tensor=True)
    if torch.cuda.is_available():
        queries_embeddings = queries_embeddings.to('cuda')
    queries_embeddings = sbert.util.normalize_embeddings(queries_embeddings)  # Normalize as we wish to get cosine_similarity

    # Get the top_k "closest" documents for each query (a single query in our case)
    hits = sbert.util.semantic_search(queries_embeddings, embeddings.paper_embeddings, score_function=sbert.util.dot_score, top_k=nb_results)

    # Return the results for the first (and only) query this function was fed with
    return hits[0]


@st.cache
def retrieve_similar_docs(query, embeddings, df, nb_results=10):
    """ Find similar documents and build the response as a dictionnary
    Args:
            query (str): the query (concatenation of a title and the abstract)
            embeddings (PaperEmbeddings): Embeddings of collection and embedding model
            df    (Pandas dataframe): the metadata for all documents in our collection
            nb_results (int): number of similar documents (neighbours) to return

    Returns:
            A list of arXiv.org documents (similarity score, url, title, abstract)
    """
    hits = find_similar_docs(query, embeddings, nb_results)
    res  = []
    for hit in hits:
        score      = hit['score']
        paper_id   = embeddings.paper_ids[hit['corpus_id']]
        paper_url  = "https://arxiv.org/abs/{}".format(paper_id.decode("utf-8"))
        paper_title, paper_abstract = get_paper_title_abstract(df, paper_id.decode("utf-8"))
        res.append({
            'score': score, 'url': paper_url,
            'title': paper_title, 'abstract': paper_abstract})
    return res

#
# Instantiate the arxiv metadata and the embeddings object
#
data_dir = "."
arxiv_data_fName       = "arxiv_data.jsonl.gz"
arxiv_embeddings_fName = "arxiv_embeddings_all.h5"

arxiv_df         = load_arxiv_meta_data(os.path.join(data_dir, arxiv_data_fName))
arxiv_embeddings = PaperEmbeddings(os.path.join(data_dir, arxiv_embeddings_fName))
