# Title:  app.py
# Author: Didier Guillevic
# Date:   2021

import streamlit as st
import pandas as pd
import numpy as np
import os
import io

import arxiv_info

st.title("Semantic Search (arxiv.org)")
st.markdown("""
* Query with some document (preferred) or some keywords (in any language).
* Get papers from [arxiv.org](https://arxiv.org) with meaning similar to the query (arxiv indexed up to 2020-11).
* Sample queries:
  * Techniques to measure blood sugar levels.
  * Aujourd'hui, malgré des décennies de développement de la médecine et l'intérêt croissant pour les soins de santé de précision, la grande majorité des diagnostics se produisent une fois que les patients commencent à montrer des signes visibles de maladie.
  * Diabetes-Patienten kontrollieren ständig den Blutzucker. Forscher haben jetzt die Wirkung von speziellen Hautsensoren geprüft, die ohne Blut auskommen.
  * 機械学習をヘルスケアに適用することで、多くの素晴らしい結果が得られました。ただし、これらの大部分は、治療に明示的にではなく、状態の診断または結果の予測に焦点を合わせています。
""")

#
# Define the form to get the query and desired number of results
#

def build_markdown_response(docs):
    outStream = io.StringIO()
    for doc in docs:
        outStream.write(f"* Score   : {doc['score']:0.3f}\n")
        outStream.write(f"* URL     : [{doc['url']}]({doc['url']})\n")
        outStream.write(f"* Title   : {doc['title']}\n")
        outStream.write(f"* Abstract: {doc['abstract']}\n")
        outStream.write("- - - -\n")
    result = outStream.getvalue()
    outStream.close()
    return result

def form_callback():
    # Get the top semantically similar documents
    docs = arxiv_info.retrieve_similar_docs(
        st.session_state.query,
        arxiv_info.arxiv_embeddings,
        arxiv_info.arxiv_df,
        nb_results=st.session_state.nb_results)

    # Return the result in markdown format
    st.header("Search document")
    st.write(st.session_state.query)
    st.header("Similar documents")
    result = build_markdown_response(docs)
    st.markdown(result)

with st.form(key="input_form"):
    st.text_area("Text to search", key="query")
    st.slider(label="Number of results", min_value=1, max_value=10, value=5, key="nb_results")
    submit_button = st.form_submit_button(label="Submit", on_click=form_callback)
