import gradio as gr
import os
import openai
import json
import tiktoken
import pandas as pd

openai.api_key = os.environ["OPENAI_API_KEY"]
prompt_templates = {"Default ChatGPT": ""}

def num_tokens_from_messages(messages, model="gpt-3.5-turbo"):
    """Returns the number of tokens used by a list of messages."""
    try:
        encoding = tiktoken.encoding_for_model(model)
    except KeyError:
        encoding = tiktoken.get_encoding("cl100k_base")
    if model == "gpt-3.5-turbo": 
        num_tokens = 0
        for message in messages:
            num_tokens += 4  # every message follows <im_start>{role/name}\n{content}<im_end>\n
            for key, value in message.items():
                num_tokens += len(encoding.encode(value))
                if key == "name":  # if there's a name, the role is omitted
                    num_tokens += -1  # role is always required and always 1 token
        num_tokens += 2  # every reply is primed with <im_start>assistant
        return num_tokens
    else:
        raise NotImplementedError(f"""num_tokens_from_messages() is not presently implemented for model {model}. 
        See https://github.com/openai/openai-python/blob/main/chatml.md for information on how messages are converted to tokens.""")


def get_empty_state():
    return {"total_tokens": 0, "messages": [], "threshold": 0}

def download_prompt_templates():
    df = pd.read_csv('prompts.csv', encoding='unicode_escape')
    prompt_templates.update(dict(zip(df['act'], df['prompt'])))
    choices = list(prompt_templates.keys())
    return gr.update(value=choices[0], choices=choices)

def on_token_change(user_token):
    openai.api_key = user_token or os.environ.get("OPENAI_API_KEY")

def on_prompt_template_change(prompt_template):
    if not isinstance(prompt_template, str): return
    return prompt_templates[prompt_template]

def submit_message(prompt, prompt_template, temperature, max_tokens, state):

    history = state['messages']

    if not prompt:
        return gr.update(value='', visible=state['total_tokens'] < 1_000), [(history[i]['content'], history[i+1]['content']) for i in range(0, len(history)-1, 2)], f"Total tokens used: {state['total_tokens']} / 4090", state
    
    prompt_template = prompt_templates[prompt_template]
    print(prompt_template)
    system_prompt = []
    if prompt_template:
        system_prompt = [{ "role": "system", "content": prompt_template}]

    prompt_msg = {"role": "user", "content": prompt }
    
    # check length token message
    messages = system_prompt + history + [prompt_msg]
    history_id = 2
    while num_tokens_from_messages(messages) >= 4090:
        messages = system_prompt + history[history_id:] + [prompt_msg]
        history_id +=2
        state['threshold'] +=1
        if history_id > len(history):
            break
    try:
        completion = openai.ChatCompletion.create(model="gpt-3.5-turbo", messages=messages, temperature=temperature, max_tokens=max_tokens)
        history.append(prompt_msg)
        history.append(completion.choices[0].message.to_dict())

        state['total_tokens'] += completion['usage']['total_tokens']
    
    except Exception as e:
        history.append(prompt_msg)
        history.append({
            "role": "system",
            "content": f"Error: {e}"
        })

    total_tokens_used_msg = f"Total tokens used: {state['total_tokens']} / 4090. "
    chat_messages = [(history[i]['content'], history[i+1]['content']) for i in range(0, len(history)-1, 2)]

    if state['threshold'] >= 3:
        input_visibility = False
        total_tokens_used_msg += "Reach the limit of this conversation. Start the new one"
    else:
        input_visibility = True

    return gr.update(value='', visible=input_visibility), chat_messages, total_tokens_used_msg, state

def clear_conversation():
    return gr.update(value=None, visible=True), None, "", get_empty_state()

css = """
      #col-container {max-width: 80%; margin-left: auto; margin-right: auto;}
      #chatbox {min-height: 400px;}
      #header {text-align: center;}
      #prompt_template_preview {padding: 1em; border-width: 1px; border-style: solid; border-color: #e0e0e0; border-radius: 4px;}
      #total_tokens_str {text-align: right; font-size: 0.8em; color: #666; height: 1em;}
      #label {font-size: 0.8em; padding: 0.5em; margin: 0;}
      """

with gr.Blocks(css=css) as demo:
    
    state = gr.State(get_empty_state())


    with gr.Column(elem_id="col-container"):
        gr.Markdown("""## OpenAI ChatGPT with awesome prompts
                    Current limit is 4090 tokens per conversation<br>
                    Input your text with a custom insruction (If need).""",
                    elem_id="header")

        with gr.Row():
            with gr.Column():
                chatbot = gr.Chatbot(elem_id="chatbox")
                input_message = gr.Textbox(show_label=False, placeholder="Enter text and press enter", visible=True).style(container=False)
                total_tokens_str = gr.Markdown(elem_id="total_tokens_str")
                btn_clear_conversation = gr.Button("🔃 Start New Conversation")
            with gr.Column():
                prompt_template = gr.Dropdown(label="Set a custom insruction for the chatbot:", choices=list(prompt_templates.keys()))
                prompt_template_preview = gr.Markdown(elem_id="prompt_template_preview")

                with gr.Accordion("Advanced parameters", open=False):
                    temperature = gr.Slider(minimum=0, maximum=2.0, value=0.7, step=0.1, interactive=True, label="Temperature (higher = more creative/chaotic)")
                    max_tokens = gr.Slider(minimum=100, maximum=4096, value=1000, step=1, interactive=True, label="Max tokens per response")

    input_message.submit(submit_message, [input_message, prompt_template, temperature, max_tokens, state], [input_message, chatbot, total_tokens_str, state])
    btn_clear_conversation.click(clear_conversation, [], [input_message, chatbot, total_tokens_str, state])
    prompt_template.change(on_prompt_template_change, inputs=[prompt_template], outputs=[prompt_template_preview])

    demo.load(download_prompt_templates, inputs=None, outputs=[prompt_template])


# demo.launch(debug=True, height='800px', auth=("admin", "dtm1234"))
demo.launch(debug=True, height='800px')