"""
The image retreival section of the project takes a text prompt as input and fetches images from dataset that are semantically close the text prompt.
1. Gradio version that is deployed providing interface in Hugging Face Space.
"""
# Authors: DeeKay Goswami & Naresh Kumar Devulapally    

import os
import json
import uuid
import time
import zipfile
import threading
import subprocess
import numpy as np
import gradio as gr
import pandas as pd
from PIL import Image
from transformers import CLIPProcessor, CLIPModel
from sklearn.metrics.pairwise import cosine_similarity

# This will load the model and processor...
model = CLIPModel.from_pretrained("openai/clip-vit-base-patch32")
processor = CLIPProcessor.from_pretrained("openai/clip-vit-base-patch32")

# This will load features from Parquet file...
parquet_feature_path = "./COCO_Features/COCO_Features.parquet"
df = pd.read_parquet(parquet_feature_path)

# This will store filenames and features from the dataframe...
all_filenames = df["filename"].tolist()
features_for_all_images = df["feature"].tolist()

image_directory = "./Dataset/"

def build_image_index(base_directory):
    image_index = {}
    for root, dirs, files in os.walk(base_directory):
        for filename in files:
            if filename.endswith('.jpg'):
                full_path = os.path.join(root, filename)
                image_index[filename] = full_path
    return image_index

image_index = build_image_index(image_directory)

with open("image_index.json", "w") as f:
    json.dump(image_index, f)

with open('image_index.json', 'r') as f:
    loaded_index = json.load(f)

def transfer_sh_upload(file_name):
    try:
        output = subprocess.check_output(["curl", "--upload-file", file_name, f"https://transfer.sh/{os.path.basename(file_name)}"])
        url = output.decode("utf-8").strip()
        url = url.replace("https://transfer.sh/", "https://transfer.sh/get/")
        return url
    
    except Exception as e:
        print(f"Error downloading file: {e}")
        return None
        
def delayed_delete(filename, delay = 500):
    time.sleep(delay)
    try:
        os.remove(filename)
    except Exception as e:
        print(f"Error deleting {filename}: {e}")

def get_image_path(filename):
    folders = os.listdir(image_directory)
    for folder in folders:
        potential_path = os.path.join(image_directory, folder, filename)
        if os.path.exists(potential_path):
            return potential_path
    raise FileNotFoundError(f"Could not locate {filename} in any subdirectory.")

def concatenate_images(images):
    images = [img if len(img.shape) == 3 and img.shape[2] == 3 else np.stack([img, img, img], axis=-1) for img in images]
    max_height = max(img.shape[0] for img in images)
    padded_images = [np.pad(img, ((0, max_height - img.shape[0]), (0, 0), (0, 0))) for img in images]
    concatenated_images = np.concatenate(padded_images, axis=1)
    
    return concatenated_images

def fetch_images(query, number):
    text = [query]
    number = int(number)
    text_inputs = processor(text=text, return_tensors="pt", padding=True, truncation=True)
    text_outputs = model.get_text_features(**text_inputs)
    text_features = text_outputs.detach()

    sim_scores = [cosine_similarity(text_features, np.array(img_feature).reshape(1, -1))[0][0] for img_feature in features_for_all_images]
    top_indices = sorted(range(len(sim_scores)), key=lambda i: sim_scores[i])[-number:]
    
    top_image_paths = [loaded_index[all_filenames[i]] for i in top_indices]
    
    zip_filename = f"images_{uuid.uuid4()}.zip"
    with zipfile.ZipFile(zip_filename, "w") as zipf:
        for img_path in top_image_paths:
            zipf.write(img_path, os.path.basename(img_path))
    url = transfer_sh_upload(zip_filename)
    download_link = f"<a href='{url}' download>Click here to download requested images</a>"
    threading.Thread(target=delayed_delete, args=(zip_filename,)).start()

    top_images_display = [np.array(Image.open(img_path)) for img_path in top_image_paths[:2]]
    while len(top_images_display) < 1:
        top_images_display.append(None)

    return (concatenate_images(top_images_display), download_link)

examples = [
    ["Surfing", "2"],
    ["Children on Picnic", "2"],
    ["Kid Playing BaseBall", "2"],
    ["Girl with the Umbrella", "2"],
    ["Girl with the Fancy Tattoo", "2"],
]

iface = gr.Interface(
    fn=fetch_images,
    inputs=[
        gr.inputs.Textbox(label="Search Query"),
        gr.inputs.Textbox(label="Number of Images", default="2"),
    ],
    outputs=[
        gr.outputs.Image(type="numpy", label="Most Similar Images"),
        gr.outputs.HTML(label="Download Link")
    ],
    examples=examples,
    title="Zero-Shot COCO-Google",
    description="Enter a query just like you Google to search images. Powered by OpenAI's Visual Transformer Model & Microsoft's COCO dataset of 1.5 Million Objects"
)

iface.launch()

