# Demo for T5 trained on multi-angular AAAC

import json
import textwrap
import re

import graphviz
import requests
import seaborn as sns
import streamlit as st 
from spacy import displacy
from stqdm import stqdm
from transformers import pipeline

import aaac_util as aaac


MODEL = "debatelab/argument-analyst"

INTRO_TEXT = """This app let's you explore ArgumentAnalyst, a system for 
logical analysis and reconstruction of argumentative texts (DeepA2). 
See also [Betz / Richardson 2021](https://arxiv.org/abs/2110.01509).

---"""

BOILER_PLATE = {
    "Enter your story" : "",
    "Enter question (optional)": "",
}

MODES = [
    # informal analysis
    {'id':'s => a','from':['argument_source'],'to':'argdown_reconstruction'},
    {'id':'s+r => a','from':['argument_source','reason_statements'],'to':'argdown_reconstruction'},
    {'id':'s+j => a','from':['argument_source','conclusion_statements'],'to':'argdown_reconstruction'},
    {'id':'r+j => a','from':['reason_statements','conclusion_statements'],'to':'argdown_reconstruction'},
    {'id':'s+r+j => a','from':['argument_source','reason_statements','conclusion_statements'],'to':'argdown_reconstruction'},
    {'id':'s => r','from':['argument_source'],'to':'reason_statements'},
    {'id':'s+a => r','from':['argument_source','argdown_reconstruction'],'to':'reason_statements'},
    {'id':'s+j => r','from':['argument_source','conclusion_statements'],'to':'reason_statements'},
    {'id':'s => j','from':['argument_source'],'to':'conclusion_statements'},
    {'id':'s+a => j','from':['argument_source','argdown_reconstruction'],'to':'conclusion_statements'},
    {'id':'s+r => j','from':['argument_source','reason_statements'],'to':'conclusion_statements'},
    # extract premises and conclusions
    {'id':'a => c','from':['argdown_reconstruction'],'to':'conclusion'},
    {'id':'a => p','from':['argdown_reconstruction'],'to':'premises'},
    # formalize
    {'id':'p => f','from':['premises'],'to':'premises_formalized'},
    {'id':'p+c+o => f','from':['premises','conclusion','conclusion_formalized'],'to':'premises_formalized'},
    {'id':'c => o','from':['conclusion'],'to':'conclusion_formalized'},
    {'id':'c+p+f => o','from':['conclusion','premises','premises_formalized'],'to':'conclusion_formalized'},
    {'id':'p+f => k','from':['premises','premises_formalized'],'to':'plcd_subs'},
    {'id':'c+o => k','from':['conclusion','conclusion_formalized'],'to':'plcd_subs'},
    {'id':'p+f+c+o => k','from':['premises','premises_formalized','conclusion','conclusion_formalized'],'to':'plcd_subs'},
    # re-reconstruct argument
    {'id':'f+k => p','from':['premises_formalized','plcd_subs'],'to':'premises'},
    {'id':'o+k => c','from':['conclusion_formalized','plcd_subs'],'to':'conclusion'},
    {'id':'p+c => a','from':['premises','conclusion'],'to':'argdown_reconstruction'}
]

TEST_DATA = [
    {"title":"Allergies (AAAC, 2 steps, 2 distractors, 1 implicit premise, 1 implicit intermediary conclusion)","argument_source":"Whoever is a sufferer of allergy to mango is not a sufferer of allergy to sesame or a sufferer of allergy to carrot. And no sufferer of allergy to carrot is hypersensitive to mango. Consequently, every sufferer of allergy to mango reacts allergically to turkey. Yet someone who is not a sufferer of allergy to mango and a sufferer of allergy to cheese is a sufferer of allergy to ginger and a sufferer of allergy to pepper. Plus, every person who is not both a sufferer of allergy to maize and a sufferer of allergy to mustard is a sufferer of allergy to cinnamon or a sufferer of allergy to oat.","argdown_reconstruction":"(1) If someone is a sufferer of allergy to mango, then they are a sufferer of allergy to carrot, or not a sufferer of allergy to sesame.\n(2) If someone is a sufferer of allergy to carrot, then they are not a sufferer of allergy to mango.\n--\nwith generalized disjunctive syllogism (transposition, negation variant) from (1), (2)\n--\n(3) If someone is a sufferer of allergy to mango, then they are not a sufferer of allergy to sesame.\n(4) If someone is a sufferer of allergy to mango, then they are a sufferer of allergy to sesame or a sufferer of allergy to turkey.\n--\nwith generalized disjunctive syllogism from (3), (4)\n--\n(5) If someone is a sufferer of allergy to mango, then they are a sufferer of allergy to turkey.","reason_statements":[{"text":"Whoever is a sufferer of allergy to mango is not a sufferer of allergy to sesame or a sufferer of allergy to carrot","starts_at":0,"ref_reco":1},{"text":"no sufferer of allergy to carrot is hypersensitive to mango","starts_at":121,"ref_reco":2}],"conclusion_statements":[{"text":"every sufferer of allergy to mango reacts allergically to turkey","starts_at":196,"ref_reco":5}],"premises":[{"ref_reco":1,"text":"If someone is a sufferer of allergy to mango, then they are a sufferer of allergy to carrot, or not a sufferer of allergy to sesame.","explicit":"true"},{"ref_reco":2,"text":"If someone is a sufferer of allergy to carrot, then they are not a sufferer of allergy to mango.","explicit":"true"},{"ref_reco":4,"text":"If someone is a sufferer of allergy to mango, then they are a sufferer of allergy to sesame or a sufferer of allergy to turkey.","explicit":"false"}],"premises_formalized":[{"form":"(x): ${F1}x -> (${F4}x v \u00ac${F2}x)","ref_reco":1},{"form":"(x): ${F4}x -> \u00ac${F1}x","ref_reco":2},{"form":"(x): ${F1}x -> (${F2}x v ${F3}x)","ref_reco":4}],"conclusion":[{"ref_reco":5,"text":"If someone is a sufferer of allergy to mango, then they are a sufferer of allergy to turkey."}],"conclusion_formalized":[{"form":"(x): ${F1}x -> ${F3}x","ref_reco":5}],"intermediary_conclusions_formalized":[{"form":"(x): ${F1}x -> \u00ac${F2}x","ref_reco":3}],"intermediary_conclusions":[{"ref_reco":3,"text":"If someone is a sufferer of allergy to mango, then they are not a sufferer of allergy to sesame."}],"distractors":["Every person who is not both a sufferer of allergy to maize and a sufferer of allergy to mustard is a sufferer of allergy to cinnamon or a sufferer of allergy to oat.","Someone who is not a sufferer of allergy to mango and a sufferer of allergy to cheese is a sufferer of allergy to ginger and a sufferer of allergy to pepper."],"id":"8c2c3329-cab8-4bd1-b4e7-3ff26506be9d","predicate_placeholders":["F1","F2","F3","F4"],"entity_placeholders":[],"steps":2,"n_premises":3,"n_distractors":2,"base_scheme_groups":["generalized disjunctive syllogism"],"scheme_variants":["transposition","negation variant"],"domain_id":"allergies","domain_type":"persons","plcd_subs":{"F1":"sufferer of allergy to mango","F2":"sufferer of allergy to sesame","F3":"sufferer of allergy to turkey","F4":"sufferer of allergy to carrot"},"argdown_index_map":{"s0c":5,"s0p0":4,"s1c":3,"s1p1":2,"s2c":2,"s2p0":1,"s1p0":1},"presentation_parameters":{"resolve_steps":[1],"direction":"forward","implicit_conclusion":"false","implicit_premise":"true","redundancy_frequency":0.1,"drop_conj_frequency":0.1,"start_sentence":[0,2]}},
    {"title":"Families (AAAC, 3 steps, 0 distractors, 1 implicit premise)","argument_source":"A person who is not a nephew of Richard is a half-brother of Lance or a son of Jeff, and vice versa. Hence, somebody who is not a nephew of Richard is a half-brother of Lance or a son of Jeff. We may conclude that nobody is neither a nephew of Richard nor a son of Jeff. All this entails that a person who is not a great-grandfather of David is a son of Jeff, owing to the fact that someone who is not a great-grandfather of David is not a nephew of Richard.","argdown_reconstruction":"(1) If, and only if, someone is not a nephew of Richard, then they are a half-brother of Lance or a son of Jeff.\n--\nwith generalized biconditional elimination (negation variant, complex variant) from (1)\n--\n(2) If someone is not a nephew of Richard, then they are a half-brother of Lance or a son of Jeff.\n(3) If someone is a half-brother of Lance, then they are a nephew of Richard.\n--\nwith generalized disjunctive syllogism (transposition, negation variant) from (2), (3)\n--\n(4) If someone is not a nephew of Richard, then they are a son of Jeff.\n(5) If someone is not a great-grandfather of David, then they are not a nephew of Richard.\n--\nwith hypothetical syllogism (negation variant) from (4), (5)\n--\n(6) If someone is not a great-grandfather of David, then they are a son of Jeff.","reason_statements":[{"text":"A person who is not a nephew of Richard is a half-brother of Lance or a son of Jeff, and vice versa","starts_at":0,"ref_reco":1},{"text":"someone who is not a great-grandfather of David is not a nephew of Richard","starts_at":383,"ref_reco":5}],"conclusion_statements":[{"text":"somebody who is not a nephew of Richard is a half-brother of Lance or a son of Jeff","starts_at":108,"ref_reco":2},{"text":"nobody is neither a nephew of Richard nor a son of Jeff","starts_at":214,"ref_reco":4},{"text":"a person who is not a great-grandfather of David is a son of Jeff","starts_at":293,"ref_reco":6}],"premises":[{"ref_reco":1,"text":"If, and only if, someone is not a nephew of Richard, then they are a half-brother of Lance or a son of Jeff.","explicit":"true"},{"ref_reco":3,"text":"If someone is a half-brother of Lance, then they are a nephew of Richard.","explicit":"false"},{"ref_reco":5,"text":"If someone is not a great-grandfather of David, then they are not a nephew of Richard.","explicit":"true"}],"premises_formalized":[{"form":"(x): \u00ac${F2}x <-> (${F4}x v ${F3}x)","ref_reco":1},{"form":"(x): ${F4}x -> ${F2}x","ref_reco":3},{"form":"(x): \u00ac${F1}x -> \u00ac${F2}x","ref_reco":5}],"conclusion":[{"ref_reco":6,"text":"If someone is not a great-grandfather of David, then they are a son of Jeff."}],"conclusion_formalized":[{"form":"(x): \u00ac${F1}x -> ${F3}x","ref_reco":6}],"intermediary_conclusions_formalized":[{"form":"(x): \u00ac${F2}x -> (${F4}x v ${F3}x)","ref_reco":2},{"form":"(x): \u00ac${F2}x -> ${F3}x","ref_reco":4}],"intermediary_conclusions":[{"ref_reco":2,"text":"If someone is not a nephew of Richard, then they are a half-brother of Lance or a son of Jeff."},{"ref_reco":4,"text":"If someone is not a nephew of Richard, then they are a son of Jeff."}],"distractors":[],"id":"5812d3a0-05d0-4e50-af62-416205f6ea22","predicate_placeholders":["F1","F2","F3","F4"],"entity_placeholders":[],"steps":3,"n_premises":3,"n_distractors":0,"base_scheme_groups":["hypothetical syllogism","generalized biconditional elimination","generalized disjunctive syllogism"],"scheme_variants":["transposition","negation variant","complex variant"],"domain_id":"male_relatives","domain_type":"persons","plcd_subs":{"F1":"great-grandfather of David","F2":"nephew of Richard","F3":"son of Jeff","F4":"half-brother of Lance"},"argdown_index_map":{"s0c":6,"s0p0":5,"s1c":4,"s1p1":3,"s2c":2,"s2p0":1},"presentation_parameters":{"resolve_steps":[],"direction":"forward","implicit_conclusion":"false","implicit_premise":"true","redundancy_frequency":0.1,"drop_conj_frequency":0.1,"start_sentence":[0,1]}},
    {"title":"Football (AAAC, 4 steps, 0 distractors, 0 implicit premises, 3 implicit intermediary conclusions, implicit final conclusion)", "argument_source":"If, and only if, someone is a critic of Besiktas JK, then they are an expert of Tottenham Hotspur or a friend of KRC Genk. And everybody who hasn't expert knowledge about Kilmarnock FC doesn't criticize Besiktas JK, and no expert of Kilmarnock FC has expert knowledge about Tottenham Hotspur, and vice versa.","argdown_reconstruction":"(1) If, and only if, someone is a critic of Besiktas JK, then they are an expert of Tottenham Hotspur or a friend of KRC Genk.\n--\nwith generalized biconditional elimination (negation variant, complex variant) from (1)\n--\n(2) If someone is a critic of Besiktas JK, then they are an expert of Tottenham Hotspur or a friend of KRC Genk.\n(3) If, and only if, someone is an expert of Kilmarnock FC, then they are not an expert of Tottenham Hotspur.\n--\nwith generalized biconditional elimination (negation variant) from (3)\n--\n(4) If someone is an expert of Kilmarnock FC, then they are not an expert of Tottenham Hotspur.\n(5) If someone is not an expert of Kilmarnock FC, then they are not a critic of Besiktas JK.\n--\nwith hypothetical syllogism (transposition, negation variant) from (4), (5)\n--\n(6) If someone is a critic of Besiktas JK, then they are not an expert of Tottenham Hotspur.\n--\nwith generalized disjunctive syllogism from (2), (6)\n--\n(7) If someone is a critic of Besiktas JK, then they are a friend of KRC Genk.","reason_statements":[{"text":"If, and only if, someone is a critic of Besiktas JK, then they are an expert of Tottenham Hotspur or a friend of KRC Genk","starts_at":0,"ref_reco":1},{"text":"everybody who hasn't expert knowledge about Kilmarnock FC doesn't criticize Besiktas JK","starts_at":127,"ref_reco":5},{"text":"no expert of Kilmarnock FC has expert knowledge about Tottenham Hotspur, and vice versa","starts_at":220,"ref_reco":3}],"conclusion_statements":[],"premises":[{"ref_reco":1,"text":"If, and only if, someone is a critic of Besiktas JK, then they are an expert of Tottenham Hotspur or a friend of KRC Genk.","explicit":"true"},{"ref_reco":3,"text":"If, and only if, someone is an expert of Kilmarnock FC, then they are not an expert of Tottenham Hotspur.","explicit":"true"},{"ref_reco":5,"text":"If someone is not an expert of Kilmarnock FC, then they are not a critic of Besiktas JK.","explicit":"true"}],"premises_formalized":[{"form":"(x): ${F1}x <-> (${F2}x v ${F3}x)","ref_reco":1},{"form":"(x): ${F4}x <-> \u00ac${F2}x","ref_reco":3},{"form":"(x): \u00ac${F4}x -> \u00ac${F1}x","ref_reco":5}],"conclusion":[{"ref_reco":7,"text":"If someone is a critic of Besiktas JK, then they are a friend of KRC Genk."}],"conclusion_formalized":[{"form":"(x): ${F1}x -> ${F3}x","ref_reco":7}],"intermediary_conclusions_formalized":[{"form":"(x): ${F1}x -> (${F2}x v ${F3}x)","ref_reco":2},{"form":"(x): ${F4}x -> \u00ac${F2}x","ref_reco":4},{"form":"(x): ${F1}x -> \u00ac${F2}x","ref_reco":6}],"intermediary_conclusions":[{"ref_reco":2,"text":"If someone is a critic of Besiktas JK, then they are an expert of Tottenham Hotspur or a friend of KRC Genk."},{"ref_reco":4,"text":"If someone is an expert of Kilmarnock FC, then they are not an expert of Tottenham Hotspur."},{"ref_reco":6,"text":"If someone is a critic of Besiktas JK, then they are not an expert of Tottenham Hotspur."}],"distractors":[],"id":"ead34d89-af68-4add-bb62-caff9043c90f","predicate_placeholders":["F1","F2","F3","F4"],"entity_placeholders":[],"steps":4,"n_premises":3,"n_distractors":0,"base_scheme_groups":["hypothetical syllogism","generalized biconditional elimination","generalized disjunctive syllogism"],"scheme_variants":["transposition","negation variant","complex variant"],"domain_id":"football_fans","domain_type":"persons","plcd_subs":{"F1":"critic of Besiktas JK","F2":"expert of Tottenham Hotspur","F3":"friend of KRC Genk","F4":"expert of Kilmarnock FC"},"argdown_index_map":{"s0c":7,"s1c":2,"s1p1":2,"s2c":6,"s2p0":5,"s1p0":1,"s3c":4,"s3p0":3},"presentation_parameters":{"resolve_steps":[1,2,3],"direction":"backward","implicit_conclusion":"true","implicit_premise":"false","redundancy_frequency":0.1,"drop_conj_frequency":0.1,"start_sentence":[0,2]}}
]

GEN_CHAINS = [
    {"id":"straight (with formalization)","modes":['s => a','s => r','s => j','a => c','a => p','c => o','p+c+o => f']},
    {"id":"straight (without formalization)","modes":['s => a','s => r','s => j']},
    {"id":"hermeneutic cycle 1","modes":['s => a','s+a => r','s+a => j','r+j => a','a => c','a => p','c => o','p+c+o => f']},
    {"id":"hermeneutic cycle 2","modes":['s => a','s+a => r','s+a => j','s+r+j => a','s+a => r','s+a => j','s+r+j => a','a => c','a => p','c => o','p+c+o => f']},
    {"id":"logical streamlining","modes":['s => a','a => p','a => c','c => o','c+o => k','o+k => c','p+c => a','a => c','a => p','c => o','p+c+o => f']},
]

INFERENCE_PARAMS = {
    'max_length':450,
    'use_cache':False
}

MAX_API_CALLS = 3

HTML_WRAPPER = """<div style="overflow-x: auto; border: 1px solid #e6e9ef; border-radius: 0.25rem; padding: 1rem; margin-bottom: 2.5rem">{}</div>"""

CACHE_SIZE = 10000

def params(config):
    pass


@st.cache(allow_output_mutation=True)
def aaac_fields():
    fields = []
    for m in MODES:
        fields = fields + m['from']
    return set(sorted(fields))

# defines how to present reason and conclusion statements to the model
@st.cache(allow_output_mutation=True)
def format_statements_list(statements: list) -> str:
    if len(statements)==0:
        return "None"
    list_as_string = ["%s (ref: (%s))" % (sdict['text'],sdict['ref_reco']) for sdict in statements]
    list_as_string = " | ".join(list_as_string)
    return list_as_string

# construct inference graph
@st.cache(allow_output_mutation=True)
def get_inference_graph(argdown_parsed,colors):
    premise_template = """<
    <TABLE BORDER="0" COLOR="#444444" CELLPADDING="10" CELLSPACING="2">
    <TR><TD BORDER="0" BGCOLOR="{bgcolor}" STYLE="rounded"><FONT FACE="sans serif" POINT-SIZE="12"><B>({label})</B> {text}</FONT></TD></TR>
    </TABLE>
    >"""

    conclusion_template = """<
    <TABLE BORDER="0" COLOR="#444444" CELLPADDING="10" CELLSPACING="2">
    <TR><TD BORDER="1" BGCOLOR="white" CELLPADDING="4"><FONT FACE="sans serif" POINT-SIZE="10">{inference}</FONT></TD></TR>
    <TR><TD BORDER="0" BGCOLOR="{bgcolor}" STYLE="rounded"><FONT FACE="sans serif" POINT-SIZE="12"><B>({label})</B> {text}</FONT></TD></TR>
    </TABLE>
    >"""

    g = graphviz.Digraph()
    g.attr(ratio="compress", size="6,10", orientation='portrait',overlay="compress")
    for item in argdown_parsed:
        text = textwrap.wrap("(X) "+item['text'], width=30)
        text='<BR/>'.join(text)[4:]
        #g.attr('node',shape='box', fillcolor='#40e0d0', style='filled')
        g.attr('node',shape='plaintext')
        if len(item['uses'])==0:
            g.node(
                'node%d'%item['label'],
                premise_template.format(text=text,label=item['label'],bgcolor=colors.get('P%d'%item['label'],'white')),
                tooltip=textwrap.fill(item['text'], width=30)
            )
        else:
            inference = "with <I>"+item['scheme']+"</I>"
            if len(item['variants'])>0:
                inference += " ("+(", ".join(item['variants']))+"):"
            inference = textwrap.wrap(inference, width=40)
            inference='<BR/>'.join(inference)
            g.node(
                'node%d'%item['label'],
                conclusion_template.format(
                    text=text,
                    label=item['label'],
                    bgcolor=colors.get('C%d'%item['label'],'white'),
                    inference=inference
                ),
                tooltip=textwrap.fill(item['text'], width=30)
            )

        for i in item['uses']:
            g.edge('node%d'%i,'node%d'%item['label'])

    return g


# get entities for displacy
@st.cache(allow_output_mutation=True)
def get_ds_entities(argument_source,s_parsed,type="reasons"):
    if type=="reasons":
        lab_templ="P%d"
        color_profile = "mako_r"
    elif type=="conclusions":
        lab_templ="C%d"
        color_profile = "rocket_r"
    else:
        return None,None

    ents = []
    colors = []
    pointer = 0
    for item in s_parsed:
        reason = item['text']
        if reason in argument_source:
            idx_start = argument_source.index(reason, pointer)
            idx_end = idx_start+len(reason)
            pointer = idx_end
            ents.append({
                "start":idx_start, "end":idx_end, "label":lab_templ%item['ref_reco']
            })

    # construct colors for reason statements
    palette = sns.color_palette(color_profile,round(3*len(ents))).as_hex()
    colors = {ent["label"]:palette[i] for i,ent in enumerate(ents)}

    return ents,colors

# format raw argdown (inserting line breaks)
@st.cache(allow_output_mutation=True)
def format_argdown(raw_argdown: str, colors=None) -> str:
    if not raw_argdown:
        return "No argument reconstruction to display."
    all_colors = {('color('+str(i+1)+')'):'white' for i in range(20)}
    if colors:
        all_colors.update({('color('+k[1:]+')'):v for k,v in colors.items()})

    def format_statement_block(s):
        r = re.sub('(\([0-9]+\))', r'<br><b><span style="background-color:{color\1}">\1</span></b>', s)
        r = r.format(**all_colors)
        return r
    format_inference_block = lambda s: "<br>--<br><i>"+s+"</i><br>--"
    split = raw_argdown.split(' -- ')
    argdown = format_statement_block(split[0])
    i=1
    while i<len(split):
        argdown = argdown + format_inference_block(split[i])
        if i<len(split)+1: 
            argdown = argdown + format_statement_block(split[i+1])
        i = i+2
    argdown = argdown[4:]# remove first linebreak
    argdown = """<div style="font-family:monospace;font-size:14px">%s</div>"""%argdown
    return argdown

# format formalization as markdown
@st.cache(allow_output_mutation=True)
def get_formalization_display(pform_parsed=None, cform_parsed=None):
    # format premises and conclusion
    premise_list = ['- `%s` (%d)'%(p['form'],p['ref_reco']) for p in pform_parsed]
    premise_list = '\n'.join(premise_list)
    conclusion = '- `%s` (%d)'%(cform_parsed[-1]['form'],cform_parsed[-1]['ref_reco'])

    # check deductive validity
    evaluator = aaac.AAACLogicEvaluator()
    scheme = [p['form'] for p in pform_parsed]
    scheme.append(cform_parsed[-1]['form'])        
    check = evaluator.check_deductive_validity(scheme)
    if check==None:
        eval_message = "(Couldn't parse formulas.)"
    else: 
        check_token = "valid" if check else "invalid"
        eval_message = "The inference from *premises* to *conclusion* is deductively **{check_token}**.".format(check_token=check_token)

    # put everything together
    display_formalization = """##### Premises:\n\n{premise_list}\n\n##### Conclusion:\n\n{conclusion}\n\n{eval_message}"""
    display_formalization = display_formalization.format(premise_list=premise_list,conclusion=conclusion,eval_message=eval_message)

    return display_formalization


def run_model(mode_set, user_input):
    """Main method for running the model 

    :param mode_set: the modes to run
    :param user_input: the user input (dict) 
    :returns: output dict
    """

    if "inference" not in st.session_state:
        with st.spinner('Initializing pipeline'):
            st.session_state.inference = pipeline(task="text2text-generation", model=MODEL)

    current_input = user_input.copy()
    output = []

    for i,mode_id in enumerate(mode_set):
        current_mode = next(m for m in MODES if m['id']==mode_id)
        with st.spinner('Generating output %d of %d with mode %s'%((i+1),len(mode_set),mode_id)):
            # construct prompt
            inquire_prompt = ""
            to_key = current_mode['to']
            for from_key in current_mode['from']:
                inquire_prompt = inquire_prompt + (f"{to_key}: {from_key}: {current_input[from_key]}")
            # inquire model
            inputs = inquire_prompt
            out = st.session_state.inference(inputs, **INFERENCE_PARAMS)
            out = out[0]['generated_text']
            # cleanup formalization
            if to_key in ['premises_formalized','conclusion_formalized']:
                out = out.replace("⁇","") 
                out = out.replace("  "," ")
                #out = out+"-Hellooo!"
        
            # write output
            output.append({
                'step':i,
                'mode':current_mode,
                'output':out,
                'prompt':inquire_prompt
            })
            # update input
            current_input[to_key] = out

    return output
      

def main():

    st.set_page_config(layout="wide")
    st.title("DeepA2 ArgumentAnalyst Demo")
    st.markdown(INTRO_TEXT, unsafe_allow_html=False)

    ## page details

    # choose example data
    ex_titles = [x['title'] for x in TEST_DATA]
    ex_titles = ['...'] + ex_titles
    ex_s = st.selectbox("1. Select an example...",ex_titles,index=0)
    ex_item = TEST_DATA[ex_titles.index(ex_s)-1]

    user_input = {}

    # Source text
    d = 'argument_source'
    filler = d
    if ex_s != '...':
        filler = ex_item[d]
        filler = filler.lower()
    user_input[d] = st.text_area(
        "...or enter a text to analyze (argument source):",filler,
        height=250
    )
    if user_input[d]==d:
        user_input[d] = ""
    else:
        user_input[d] = user_input[d].lower()


    # modes

    gen_chain_id_s = st.selectbox("2. Select a reconstruction strategy...",[x['id'] for x in GEN_CHAINS],index=0)
    modes_s = next(x['modes'] for x in GEN_CHAINS if x['id']==gen_chain_id_s)

    modes_s = st.multiselect(
        "... or build a custom generative chain (argument source=`s`, reasons=`r`, conjectures=`j`, argdown reconstruction=`a`, premises=`p`, conclusion=`c`, premise formalization=`f`, conclusion formalization=`o`, keys=`k`):",
        [m['id'] for m in MODES]*2,
        modes_s#["s => a","s+a => r","s+a => j","a => c","a => p","c => o","p+c+o => f"]
    )

    # optional additional input 

    input_expander = st.expander(label='Additional input (optional)')
    with input_expander:
        # for every mode, add input field
        for d in [m for m in aaac_fields() if m!="argument_source"]:
            filler = d
            if ex_s != '...':
                filler = ex_item[d]
                if d in ['reason_statements','conclusion_statements','conclusion','premises']:
                    filler = aaac.AAACLayouter.format_statements_list(filler)
                elif d in ['premises_formalized','conclusion_formalized']:
                    filler = aaac.AAACLayouter.format_formalizations_list(filler)
                elif d in ['plcd_subs']:
                    filler = aaac.AAACLayouter.format_plcd_subs(filler)

            user_input[d] = st.text_area(
                d,filler,
                height=250
            )

            if user_input[d]==d:
                user_input[d] = ""


    ## answer a query
    submit = st.button("Process")

    #row = []; index = []
    #row_der = []; index_der = []


    if submit:
        if all(len(v)==0 for v in user_input.values()): 
          st.warning('Please choose an example, or provide input.')
        else:
            with st.spinner("Processing (may take a couple of minutes) ..."):
                output = run_model(modes_s,user_input)

                if output is None:
                    return None

                # get latest generated reasons, conclusions, argdown 
                argdown_raw = [out['output'] for out in output if out['mode']['to']=='argdown_reconstruction']
                argdown_raw = argdown_raw[-1] if len(argdown_raw)>0 else None
                reasons_raw = [out['output'] for out in output if out['mode']['to']=='reason_statements']
                reasons_raw = reasons_raw[-1] if len(reasons_raw)>0 else None
                concl_raw = [out['output'] for out in output if out['mode']['to']=='conclusion_statements']
                concl_raw = concl_raw[-1] if len(concl_raw)>0 else None
                pform_raw = [out['output'] for out in output if out['mode']['to']=='premises_formalized']
                pform_raw = pform_raw[-1] if len(pform_raw)>0 else None
                #pform_raw = "(x): F x -> (G x v H x) (ref: (1)) | (x): F x -> not G x (ref: (3))" # TEST
                cform_raw = [out['output'] for out in output if out['mode']['to']=='conclusion_formalized']
                cform_raw = cform_raw[-1] if len(cform_raw)>0 else None
                #cform_raw = "(x): F x -> H x (ref: (4))" # TEST

                # parse raw output
                argdown_parsed = aaac.AAACParser.parse_argdown_block(argdown_raw) if argdown_raw else None
                reasons_parsed = aaac.AAACParser.parse_statements(reasons_raw) if reasons_raw else None
                concl_parsed = aaac.AAACParser.parse_statements(concl_raw) if concl_raw else None            
                pform_parsed = aaac.AAACParser.parse_formalizations(pform_raw) if pform_raw else None     
                cform_parsed = aaac.AAACParser.parse_formalizations(cform_raw) if cform_raw else None     


                # check syntactic validity
                argdown_valid = (
                    aaac.ad_valid_syntax(argdown_parsed) &
                    aaac.ad_last_st_concl(argdown_parsed) &
                    aaac.used_prem_exist(argdown_parsed) #& 
                    #1-aaac.prem_non_used(argdown_parsed)
                ) if argdown_parsed else False
                reasons_valid = (
                    aaac.s_ord_me_subsseq(reasons_parsed,user_input['argument_source'])
                ) if reasons_parsed else False
                concl_valid = (
                    aaac.s_ord_me_subsseq(concl_parsed,user_input['argument_source']) 
                ) if concl_parsed else False
                reasons_concl_mc = (
                    aaac.reason_concl_mutually_exclusive(reasons_parsed,concl_parsed)
                ) if reasons_parsed and concl_parsed else False
                pform_valid = True if pform_parsed else False
                cform_valid = True if cform_parsed else False

                # get and merge entities and colors for displacy
                ents = []
                colors={}
                if concl_valid:
                    ents,colors = get_ds_entities(user_input['argument_source'],concl_parsed,type="conclusions")
                    if reasons_valid and reasons_concl_mc:                
                        ents_r,colors_r = get_ds_entities(user_input['argument_source'],reasons_parsed,type="reasons")
                        ents= ents+ents_r
                        colors.update(colors_r)
                        ents = sorted(ents, key=lambda item: item["start"])
                elif reasons_valid:
                    ents,colors = get_ds_entities(user_input['argument_source'],reasons_parsed,type="reasons")

                options = {"colors":colors}
                ex = [{"text": user_input['argument_source'],"ents": ents,"title": None}]
                displacy_html = displacy.render(ex, style="ent", options=options, manual=True)

                graphviz_graph = get_inference_graph(argdown_parsed,colors) if argdown_valid else None


            # Show output 
            col_source, col_reco = st.columns(2)
            with col_source:
                st.markdown(f'<div style="font-size: small">Reasons and conclusions in source text</div>',unsafe_allow_html=True)
                st.write(HTML_WRAPPER.format(displacy_html), unsafe_allow_html=True)

            with col_reco:
                ig_expander = st.expander(label='Argument reconstruction (inference graph)', expanded=argdown_valid)
                with ig_expander:
                    if argdown_valid:
                        st.graphviz_chart(graphviz_graph,use_container_width=True)
                    else:
                        st.write("No inference graph to display.")
                lgc_expander = st.expander(label='Formalization', expanded=(pform_valid and cform_valid))
                with lgc_expander:
                    if pform_valid and cform_valid:
                        st.markdown(get_formalization_display(pform_parsed=pform_parsed, cform_parsed=cform_parsed))
                    else:
                        st.write("No formalization to display.")
                ad_expander = st.expander(label='Argument reconstruction (argdown snippet)', expanded=(not argdown_valid))
                with ad_expander:
                    st.write(format_argdown(argdown_raw,colors), unsafe_allow_html=True)

            history_expander = st.expander(label='Full history', expanded=False)
            with history_expander:
                st.write(output)

                #for out in output:
                #    st.write("step: %d, mode: %s" % (out['step'],out['mode']['id']))
                #    output_f = format_argdown(out['output']) if out['mode']['to']=='argdown_reconstruction' else out['output']
                #    st.write(output_f, unsafe_allow_html=True)

        
if __name__ == '__main__':
    main()
