import matplotlib.pyplot as plt
import io
from PIL import Image
import pickle
import pandas as pd
import gradio as gr

def generar_recomendacion(svd_model, user_id, df, genres, top=5):
    # Filtrar las películas que correspondan al usuario y a los géneros de interés
    df_filtered = df[(df['user_id'] == user_id) & df[genres].any(axis=1)]

    # Crear un mapeo de id de película a título de película para una búsqueda más eficiente
    id_to_title = df_filtered.set_index('id')['title'].to_dict()

    # Obtener las recomendaciones utilizando la función `predict` del modelo SVD
    recommended_movies = []
    for movie_id in df_filtered['id'].unique():
        predicted_rating = svd_model.predict(user_id, movie_id).est
        recommended_movies.append((movie_id, predicted_rating))

    # Ordenar las películas según su predicción de rating
    recommended_movies.sort(key=lambda x: x[1], reverse=True)

    # Obtener los títulos de las películas recomendadas
    recommended_titles = [id_to_title[movie_id] for movie_id, _ in recommended_movies[:top]]

    # Contar cuántas películas de cada género hay en las recomendaciones
    recommended_movies_ids = [movie_id for movie_id, _ in recommended_movies[:top]]
    genre_counts = df_filtered[df_filtered['id'].isin(recommended_movies_ids)][genres].sum()

    # Limpiar la figura
    plt.clf()

    # Asignar colores específicos a cada género
    genre_colors = {'Drama': 'blue', 'Comedy': 'orange', 'Horror': 'red', 'Romance': 'pink'}
    colors = [genre_colors[genre] for genre in genres]

  
    
    # Crear el gráfico de barras con los colores específicos
    plt.style.use('ggplot')  # establece el estilo del gráfico
    plt.bar(genres, genre_counts, color=colors)
    plt.xlabel('Género', fontsize=10)
    plt.ylabel('Cantidad', fontsize=10)
    plt.title('Cantidad de Películas por Género en las Recomendaciones', fontsize=12)
    plt.grid(True)  # agrega una cuadrícula
    plt.xticks(fontsize=10)  # ajusta el tamaño de la fuente de los ticks del eje x
    plt.yticks(fontsize=10)  # ajusta el tamaño de la fuente de los ticks del eje y

    

    # Guardar el gráfico como una imagen PNG en una cadena de bytes
    buf = io.BytesIO()
    plt.savefig(buf, format='png')
    buf.seek(0)

    # Convertir la cadena de bytes en una imagen que se puede mostrar en Gradio
    im = Image.open(buf)
    im = im.convert('RGB')
    buf.close()

    # Devolver la lista de títulos y el gráfico como una imagen
    return ', '.join(recommended_titles), im


# Leer los datos
dfmerge = pd.read_csv('merged_data7.csv')

# Cargar el modelo
with open('fc_model_svd_v2.pkl', 'rb') as file:
    svd_model = pickle.load(file)

# Modificar la función wrap_generar_recomendacion para devolver una imagen también
def wrap_generar_recomendacion(user_id, drama, comedy, horror, romance, top=5):
    # Crear la lista de géneros de interés a partir de las casillas de verificación
    genres = []
    if drama: genres.append('Drama')
    if comedy: genres.append('Comedy')
    if horror: genres.append('Horror')
    if romance: genres.append('Romance')
    
    # Llamar a la función de recomendación y devolver los resultados como una cadena y una imagen
    return generar_recomendacion(svd_model, user_id, dfmerge, genres, int(top))

# Modificar la interfaz de Gradio para mostrar una imagen también
demo = gr.Interface(
    fn=wrap_generar_recomendacion,
    inputs=[gr.inputs.Number(label="User ID"), gr.inputs.Checkbox(label="Drama"), gr.inputs.Checkbox(label="Comedy"), gr.inputs.Checkbox(label="Horror"), gr.inputs.Checkbox(label="Romance"), gr.inputs.Number(label="Top")], 
    outputs=[gr.outputs.Textbox(), gr.outputs.Image(type='pil')],
    title = '<h1 style="text-align: center; color: #FF6347;">STREAMREC</h1>',
    description = """
<p>
<center>
<font size="4" face="Arial" color="white">
Sistema de Recomendaciones Personalizadas de Películas y Series
</font>
<p><b style="color: #DC143C;">Advertencia: Ingresa el ID del usuario (user_id), selecciona los géneros de interés y la cantidad de recomendaciones que te gustaría generar.
Te mostraremos algunas películas que pueden gustarte.</b></p>
<img src="https://i.pinimg.com/564x/18/51/c8/1851c8a1adbf68564f3a29e1c5c602a0.jpg" alt="logo" width="250"/>
<img src="https://i.pinimg.com/564x/22/19/69/221969071884e659af16c78455e3afde.jpg" alt="logo" width="1000" height="200"/>
</center>
</p>
""",
    
    allow_flagging='auto',
    theme="huggingface",  # establece un tema predefinido
    favicon="https://iconos8.es/icon/OrZ75sWwdNU2/comedia",  # establece tu favicon personalizado

)
# Lanzar la interfaz
demo.launch()