import gradio as gr
import pandas as pd

languages = pd.read_csv("model_lang.csv", names=["Lang_acr"])


def check_lang(lang_acronym):
    if lang_acronym in languages["Lang_acr"].to_list():
        return "True"
    else: 
        return "False"
    
title = "DReAM"

description_main = """
This space allows you to test a set of LLMs tuned to perform different tasks over dream reports. 
Three main tasks are available: 

- Name Entity Recognition (NER), with an English-only model that generates the identified characters. 

- Sentiment Analysis (SA), with two English-only models (one for multi-label classification, and one for generation) and a large multilingual model for multi-label classification.

- Relation Extraction (RE), with an English-only model that identifies relevant characters and existing relations between them following the Activity feature of the Hall and Van de Castle framework.

All models have been tuned on the Hall and Van de Castle framework. More details are on the page for each model. For more on the training framework, see the [Bertolini et al., 2023](https://arxiv.org/pdf/2302.14828.pdf) preprint.

Use the current interface to check if a language is included in the multilingual SA model, using language acronyms (e.g. it for Italian). the tabs above will direct you to each model to query.

If you want to use the models outside the space, you can easily do so via [DReAMy](https://github.com/lorenzoscottb/DReAMy)
"""

description_L = """
This model is an XLM-R tuned model, pre-trained with 94 languages available, and tuned on emotion-annotated DreamBank English data. (see original model [card](https://huggingface.co/xlm-roberta-large) to see which are available)
"""

description_S = """
A BERT-base-cased model pre-trained on English-only text and tuned on annotated DreamBank English data.
"""

description_G = """
A T5 model tuned to perform text generation and predict emotion as well as the character experiencing those emotions.
"""

description_R = """
A T5 model tuned to perform text generation and predicts the characters and the (activity) relation between them.
"""

description_GNER = """
A T5 model tuned to perform text generation, and predict which characters are present in the report. Note that, in the Hall and Van de Castle, the character lists never includes the dreamer. Hence, if you (willingly or not) enter a report that does not contain another character reference, the model will/should (correctly) produce an empty string. Moreover, it is likely that the produced list of CHAR could be longer than the one produced by the SA model, as not all CHAR might be associated with emotions. 
"""

example_main = ["en", "it", "pl"]

examples = [
    ["I was followed by the blue monster but was not scared. I was calm and relaxed."],
    ["Ero seguito dal mostro blu, ma non ero spaventato. Ero calmo e rilassato."],
    ["Śledził mnie niebieski potwór, ale się nie bałem. Byłem spokojny i zrelaksowany."],
]

examples_g = [
    ["I'm in an auditorium. Susie S is concerned at her part in this disability awareness spoof we are preparing. I ask, 'Why not do it? Lots of AB's represent us in a patronizing way. Why shouldn't we represent ourselves in a good, funny way?' I watch the video we all made. It is funny. I try to sit on a folding chair. Some guy in front talks to me. Merle is in the audience somewhere. [BL]"],

]

examples_re = [
    ["I was skating on the outdoor ice pond that used to be across the street from my house. I was not alone, but I did not recognize any of the other people who were skating around. I went through my whole repertoire of jumps, spires, and steps-some of which I can do and some of which I'm not yet sure of. They were all executed flawlessly-some I repeated, some I did only once. I seemed to know that if I went into competition, I would be sure of coming in third because there were only three contestants. Up to that time I hadn't considered it because I hadn't thought I was good enough, but now since everything was going so well, I decided to enter."],
    ["I was talking on the telephone to the father of an old friend of mine (boy, 21 years old). We were discussing the party the Saturday night before to which I had invited his son as a guest. I asked him if his son had a good time at the party. He told me not to tell his son that he had told me, but that he had had a good time, except he was a little surprised that I had acted the way I did."], 
    ["I was walking alone with my dog in a forest."]
]

interface_words = gr.Interface(
            fn=check_lang,
            inputs="text", 
            outputs="text",
            title=title,
            description=description_main,
            examples=example_main,
)

interface_model_L = gr.Interface.load(
            name="models/DReAMy-lib/xlm-roberta-large-DreamBank-emotion-presence",
            description=description_L,
            examples=examples,
            title="SA Large Multilingual",
)

interface_model_S = gr.Interface.load(
            name="models/DReAMy-lib/bert-base-cased-DreamBank-emotion-presence",
            description=description_S,
            examples=examples[0],
            title="SA Base English-Only",
)

interface_model_G = gr.Interface.load(
            name="models/DReAMy-lib/t5-base-DreamBank-Generation-Emot-Char",
            description=description_G,
            examples=examples_g,
            title="SA Generation",
)

interface_model_RE = gr.Interface.load(
            name="models/DReAMy-lib/t5-base-DreamBank-Generation-Act-Char",
            description=description_R,
            examples=examples_re,
            title="RE Generation",
)

interface_model_NER = gr.Interface.load(
            name="models/DReAMy-lib/t5-base-DreamBank-Generation-NER-Char",
            description=description_GNER,
            examples=examples_g,
            title="NER Generation",
)

gr.TabbedInterface(
    [interface_words, interface_model_NER, interface_model_L, interface_model_S, interface_model_G, interface_model_RE], 
    ["Main", "NER Generation", "SA Large Multilingual", "SA Base En", "SA En Generation", "RE Generation"]
).launch()
