import gradio as gr
from transformers import ViTFeatureExtractor, ViTForImageClassification
import numpy as np
import datetime

# Mapeamento de classe ID para rótulo
id2label = {
    "0": "dyed-lifted-polyps",
    "1": "dyed-resection-margins",
    "2": "esophagitis",
    "3": "normal-cecum",
    "4": "normal-pylorus",
    "5": "normal-z-line",
    "6": "polyps",
    "7": "ulcerative-colitis"
}

# Carregue o modelo ViT
model_name = "mrm8488/vit-base-patch16-224_finetuned-kvasirv2-colonoscopy"
feature_extractor = ViTFeatureExtractor.from_pretrained(model_name)
model = ViTForImageClassification.from_pretrained(model_name)

# Função para classificar a imagem
def classify_image(input_image):
    # Pré-processar a imagem usando o extrator de características
    inputs = feature_extractor(input_image, return_tensors="pt")
    # Realizar inferência com o modelo
    outputs = model(**inputs)
    # Obter a classe prevista
    predicted_class_id = np.argmax(outputs.logits[0].detach().numpy())
    # Obter o rótulo da classe a partir do mapeamento id2label
    predicted_class_label = id2label.get(str(predicted_class_id), "Desconhecido")
    
    # Obter a data e hora atual
    current_time = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    # Formatar a saída em HTML com rótulo da classe e data/hora
    result_html = f"""
    <h2>Resultado da Classificação</h2>
    <p><strong>Rótulo da Classe:</strong> {predicted_class_label}</p>
    <p><strong>Data e Hora:</strong> {current_time}</p>
    """
    
    # Retornar o resultado formatado em HTML
    return result_html

# Informações de como usar o aplicativo em HTML
instructions_html = """
<h2>Como Usar o Aplicativo</h2>
<ol>
<li>Clique no botão 'Escolher Arquivo' para fazer o upload de uma imagem colonoscópica.</li>
<li>Aguarde a classificação automática.</li>
<li>O resultado mostrará o rótulo da classe e a data e hora da classificação.</li>
</ol>
"""

# Criar uma interface Gradio com informações de diagnóstico, HTML e instruções
interface = gr.Interface(
    fn=classify_image,
    inputs=gr.inputs.Image(type="numpy", label="Carregar uma imagem"),
    outputs=gr.outputs.HTML(),
    title="Classificador de Imagem ViT para Colonoscopia",
    description="""
    <h3>Classifique imagens colonoscópicas usando um modelo Vision Transformer (ViT).</h3>
    <p>O modelo identificará a condição ou diagnóstico da imagem, como 'polyps', 'esophagitis', etc.</p>
    """,
    article=instructions_html
)

# Iniciar a aplicação Gradio
interface.launch(share=True)  # Compartilhar a interface com um link público
