import os
os.environ["CUDA_DEVICE_ORDER"] = "PCI_BUS_ID"

import gradio as gr
import torch
import cv2
import numpy as np
from preprocess import unsharp_masking
import glob
import time
from sklearn.cluster import KMeans  # Import K-means clustering

device = "cuda" if torch.cuda.is_available() else "cpu"
model_paths = {
    'SE-RegUNet 4GF': './model/SERegUNet4GF.pt',
    'SE-RegUNet 16GF': './model/SERegUNet16GF.pt',
    'AngioNet': './model/AngioNet.pt',
    'EffUNet++ B5': './model/EffUNetppb5.pt',
    'Reg-SA-UNet++': './model/RegSAUnetpp.pt',
    'UNet3+': './model/UNet3plus.pt',
}

print("torch: ", torch.__version__)

def filesort(img, model):
    ori = img.copy()
    img = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)
    h, w = img.shape
    img_out = preprocessing(img, model)
    return img_out, h, w, img, ori

def preprocessing(img, model='SE-RegUNet 4GF'):
    img = cv2.resize(img, (512, 512))
    img = unsharp_masking(img).astype(np.uint8)
    if model == 'AngioNet' or model == 'UNet3+':
        img = np.float32((img - img.min()) / (img.max() - img.min() + 1e-6))
        img_out = np.expand_dims(img, axis=0)
    elif model == 'SE-RegUNet 4GF':
        clahe1 = cv2.createCLAHE(clipLimit=2.0, tileGridSize=(8, 8))
        clahe2 = cv2.createCLAHE(clipLimit=8.0, tileGridSize=(8, 8))
        image1 = clahe1.apply(img)
        image2 = clahe2.apply(img)
        img = np.float32((img - img.min()) / (img.max() - img.min() + 1e-6))
        image1 = np.float32((image1 - image1.min()) / (image1.max() - image1.min() + 1e-6))
        image2 = np.float32((image2 - image2.min()) / (image2.max() - image2.min() + 1e-6))
        img_out = np.stack((img, image1, image2), axis=0)
    else:
        clahe1 = cv2.createCLAHE(clipLimit=2.0, tileGridSize=(8, 8))
        image1 = clahe1.apply(img)
        image1 = np.float32((image1 - image1.min()) / (image1.max() - image1.min() + 1e-6))
        img_out = np.stack((image1,) * 3, axis=0)
    return img_out

def process_input_image(img, model):
    pipe = torch.jit.load(model_paths[model])
    pipe = pipe.to(device).eval()
    start = time.time()
    img, h, w, ori_gray, ori = filesort(img, model)
    img = torch.FloatTensor(img).unsqueeze(0).to(device)
    with torch.no_grad():
        if model == 'AngioNet':
            img = torch.cat([img, img], dim=0)
        logit = np.round(torch.softmax(pipe.forward(img), dim=1).detach().cpu().numpy()[0, 0]).astype(np.uint8)
    spent = time.time() - start
    spent = f"{spent:.3f} second(s)"

    if h != 512 or w != 512:
        logit = cv2.resize(logit, (h, w))

    logit = logit.astype(bool)
    img_out = ori.copy()

    # Change the color of the segmented mask to red
    img_out[logit, :] = [255, 0, 0]  # Red color for the mask

    # Add a white border to the mask
    contours, _ = cv2.findContours(logit.astype(np.uint8), cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
    cv2.drawContours(img_out, contours, -1, [255, 255, 255], 2)  # White color for the border

    # Perform K-means clustering on the segmented mask
    masked_image = ori_gray.copy()
    masked_image[~logit] = 0  # Set non-segmented regions to 0
    flattened_masked_image = masked_image.reshape((-1, 1))

    # You can adjust the number of clusters (n_clusters) based on your requirements
    n_clusters = 2
    kmeans = KMeans(n_clusters=n_clusters, random_state=0).fit(flattened_masked_image)
    cluster_labels = kmeans.labels_

    # Determine the potential for anomalies based on the cluster centroids
    cluster_centers = kmeans.cluster_centers_
    anomaly_potential = np.abs(cluster_centers[0] - cluster_centers[1])

    # Define a higher threshold for classifying anomalies
    anomaly_threshold = 50  # Adjust this threshold as needed for higher rigor

    # Check if anomaly potential is above the threshold
    is_anomaly = np.sum(anomaly_potential) > anomaly_threshold

    # Provide a detailed message for cardiologists only when there's high confidence
    if is_anomaly:
        anomaly_label = "Potential Anomaly Detected: Consult a Cardiologist for Further Assessment and Diagnosis."
    else:
        anomaly_label = "No Potential Anomaly Detected. Continue Routine Cardiac Assessment."

    return spent, img_out, anomaly_label

with gr.Column():
    time_spent = gr.Label(label="Time Spent (Preprocessing + Inference)")
    img_output = gr.Image(label="Output Mask")
    anomaly_label = gr.Label(label="Anomaly Status")

my_app = gr.Blocks()
with my_app:
    gr.Markdown("Coronary Angiogram Segmentation with Gradio.")
    with gr.Tabs():
        with gr.TabItem("Select your image"):
            with gr.Row():
                with gr.Column():
                    img_source = gr.Image(label="Please select angiogram.", value='./example/angio.png', shape=(512, 512))
                    model_choice = gr.Dropdown(['SE-RegUNet 4GF', 'SE-RegUNet 16GF', 'AngioNet', 'EffUNet++ B5', 
                                                'Reg-SA-UNet++', 'UNet3+'], label='Model', info='Which model to infer?')
                    source_image_loader = gr.Button("Vessel Segment")
                with gr.Column():
                    time_spent = gr.Label(label="Time Spent (Preprocessing + Inference)")
                    img_output = gr.Image(label="Output Mask")
                    anomaly_label = gr.Label(label="Anomaly Status")

        source_image_loader.click(
            process_input_image,
            [
                img_source,
                model_choice
            ],
            [
                time_spent,
                img_output,
                anomaly_label  # Display the anomaly status label
            ]
        )

my_app.launch(debug=True)
