import requests
import ast

# Model is stored in an external Gist file, so we just need to import it.
gist_raw_url = "https://gist.githubusercontent.com/CrAvila/235330d5ce6d5a7cbdffd8de45de7630/raw"

response = requests.get(gist_raw_url)
if response.status_code == 200:
    gist_content = response.text
    gist_content = gist_content.strip()
    model_content = gist_content[len("model="):]

    # Use ast.literal_eval to safely evaluate the content as a dictionary
    try:
        model = ast.literal_eval(model_content)
    except (ValueError, SyntaxError):
        print("Error while evaluating Gist content as a dictionary.")
else:
    print("Failed to fetch Gist content.")

import numpy as np

def h_theta(theta, x):
    h = 1 / (1 + np.exp(-theta.T @ x))
    return h

def single_entry_classifier(sample, classifiers):
    sample = np.hstack((1, sample))
    predictions = []

    for clf in classifiers:
        c = np.array(clf["theta"])
        h = h_theta(c, sample)
        predictions.append((clf['target'], h))

    # Sort predictions by probability in descending order
    predictions.sort(key=lambda x: x[1], reverse=True)

    # Print the prediction
    predicted_class = predictions[0][0]

    return predicted_class

import gradio as gr
import numpy as np

def rescale(image, dim):
    new_array = []

    original_h, original_w = image.shape[:2]  # Get the original height and width

    new_h, new_w = dim, dim

    rescale_factor_w = original_w // new_w  # Calculate the scaling factors
    rescale_factor_h = original_h // new_h

    for i in range(new_h):  # Iterate over the new dimensions
        for j in range(new_w):
            start_w = int(j * rescale_factor_w)  # Calculate the start and end coordinates
            end_w = int((j + 1) * rescale_factor_w)
            start_h = int(i * rescale_factor_h)
            end_h = int((i + 1) * rescale_factor_h)

            block = image[start_h:end_h, start_w:end_w]

            new_array.append(np.mean(block))

    return new_array


# Define the function that will be called when the user draws on the canvas
def get_intensity_vector(canvas):
    # Gradio passes the canvas as a NumPy array with values ranging from 0 to 255.
    # We will normalize it to binary values where 1 represents black and 0 represents white.
    binary_image = (canvas < 128).astype(int)

    # Flatten the 2D array into a 1D vector
    intensity_vector = binary_image.flatten().tolist()

    n_vector = []
    for x in intensity_vector:
        y = 16 if x == 0 else 0
        n_vector.append(y)

    n_vector = np.array(n_vector).reshape(512, 512)

    resized_image = rescale(n_vector, 8)

    return single_entry_classifier(resized_image, model)



# Create the Gradio interface with a sketchpad input component
sp = gr.Sketchpad(shape=(512, 512), label="Digit")
output_label = "Prediction Result"
output = gr.Text(label=output_label)
iface = gr.Interface(
    fn=get_intensity_vector, 
    inputs=sp, 
    outputs=output, 
    live=True, 
    title="Digit Classifier (Logistic Regression)", 
)

# Launch the Gradio interface
iface.launch()
