import os
import requests
import gradio as gr
import torch
from transformers import ViTForImageClassification, ViTImageProcessor
from dotenv import load_dotenv
from bs4 import BeautifulSoup
from PIL import Image

load_dotenv()

# Backup
classes = ['amel', 'charcoal', 'diffused', 'cinder', 'sunkissed', 'kastanie', 'motley', 'anery', 'bloodred',
           'tessera', 'caramel', 'ghost', 'hypo', 'stripe', 'lava', 'miami', 'honey', 'snow', 'ultramel', 'wild-type']

classes = ['amelanistic', 'charcoal', 'diffused', 'cinder', 'sunkissed', 'kastanie', 'motley', 'anerythistic',
           'bloodred',
           'tessera', 'caramel', 'ghost', 'hypomelanistic', 'stripe', 'lava', 'miami', 'honey', 'snow', 'ultra',
           'wildtype']

selection = list(classes)
selection.append("None")

model = ViTForImageClassification.from_pretrained("CornSnakeID/CornSnakes", num_labels=len(classes),
                                                  problem_type="multi_label_classification",
                                                  use_auth_token=os.getenv("HUGGINGFACE_TOKEN"))

feature_extractor = ViTImageProcessor.from_pretrained("CornSnakeID/CornSnakes",
                                                      use_auth_token=os.getenv("HUGGINGFACE_TOKEN"))

prediction = {}


def classify(img):
    # Center crop to largest square
    width, height = img.size
    if width > height:
        left = (width - height) / 2
        right = width - left
        top = 0
        bottom = height
    else:
        top = (height - width) / 2
        bottom = height - top
        left = 0
        right = width
    img = img.crop((left, top, right, bottom))
    model.eval()
    with torch.inference_mode():
        outputs = model(**feature_extractor(img, return_tensors="pt"))
        probs = torch.sigmoid(outputs.logits)
    predictions = {}
    max_prob = 0
    max_class = ""
    for i, prob in enumerate(probs[0]):
        predictions[classes[i]] = prob.item()
        if prob > max_prob:
            max_prob = prob
            max_class = classes[i]

    global prediction
    prediction = dict(predictions)

    sum_text = "Looks like the estimation isn't very confident. Try another picture or change the crop."
    if max_prob > 0.5:
        sum_text = f"This snake might be a {max_class}, but you should try with more pictures or use the explore tab " \
                   f"to find out more."

    if max_prob > 0.8:
        sum_text = f"This snake is probably a {max_class}."

    if max_prob > 0.9:
        sum_text = f"This snake is very likely a {max_class}."

    return "### " + sum_text, predictions


def search(morphs):
    for i, morph in enumerate(morphs):
        morphs[i] = morphs[i].lower()

    images = []
    links = []

    html = requests.get("https://iansvivarium.com/morphs/").text
    soup = BeautifulSoup(html, "html.parser")
    topic_list = soup.find("ul", {"class": "topiclist"})

    found = 0

    for li in topic_list.find_all("li"):
        if li.find("span") is not None:
            combo = li.find("span").text.lower()
            if all(morph in combo for morph in morphs):
                src = "https://iansvivarium.com/morphs/" + li.find("img")["src"]
                src = src.replace("tiny", "large")
                # Download image
                img = requests.get(src, stream=True).raw
                img = Image.open(img).convert("RGB")

                href = f"[{combo}]({'https://iansvivarium.com/morphs/' + li.find('a')['href'].split('&')[0]})"

                images.append(img)
                links.append(href)

                found += 1

                if found >= 6:
                    break

    if len(images) > 0:
        return images, "\n".join(links)

    return None, "## No morph found."


def explore(number):
    if len(prediction) == 0 or prediction is None:
        return None, "## No morph found. Did you classify a snake?"

    max_prob = 0
    max_classes = []
    for i, p in enumerate(prediction.keys()):
        if prediction[p] > max_prob:
            max_prob = prediction[p]
            max_classes.append(p)
            if len(max_classes) > number:
                max_classes.pop(0)

    return search(max_classes)


def man_explore(m1, m2, m3):
    morphs = []
    if m1 and m1 != "None":
        morphs.append(m1)
    if m2 and m2 != "None":
        morphs.append(m2)
    if m3 and m3 != "None":
        morphs.append(m3)

    if len(morphs) == 0:
        return None, "No morph selected. Please select at least one morph to explore."
    return search(morphs)


# TODO: make this look better
css = """
.gradio-container {
background: linear-gradient(243deg, #e49f2c, #b47d21, #dfc7a0, #e4772d, #f2e1ce, #ee7f41);
background-size: 1200% 1200%;

-webkit-animation: AnimationName 0s ease infinite;
-moz-animation: AnimationName 0s ease infinite;
animation: AnimationName 0s ease infinite;
}

@-webkit-keyframes AnimationName {
    0%{background-position:0% 50%}
    50%{background-position:100% 50%}
    100%{background-position:0% 50%}
}
@-moz-keyframes AnimationName {
    0%{background-position:0% 50%}
    50%{background-position:100% 50%}
    100%{background-position:0% 50%}
}
@keyframes AnimationName {
    0%{background-position:0% 50%}
    50%{background-position:100% 50%}
    100%{background-position:0% 50%}
}"""

with gr.Blocks(analytics_enabled=True, title="Corn Snake Morph ID") as demo:
    gr.Markdown("## Corn Snake Morph Type Classifier")
    with gr.Row():
        with gr.Column():
            gr.Markdown("### Upload your photo of a corn snake")
            inp = gr.Image(shape=(224, 224), source="upload", type="pil")
            submit = gr.Button("Submit", variant="primary")
            gr.Markdown(
                "For best results, crop to a square covering the snake's body. Upload a photo taken in a cage or on a "
                "hand. Please run this model multiple times from different angles and with different lighting "
                "conditions. Correct predictions usually have a score of 85% or higher. Note that the image will be "
                "center cropped if it is not square. You may use the 'select' tool to crop the image to a square.")

        with gr.Column():
            gr.Markdown("### Results")
            summary = gr.Markdown("")
            classification = gr.Label(num_top_classes=8)
        submit.click(fn=classify, inputs=inp, outputs=[summary, classification])

    gr.Markdown("### Find out more")
    with gr.Row():
        with gr.Accordion("Explore morph images (from iansvivarium)", open=False):
            with gr.Tab("From prediction"):
                num_morphs = gr.Slider(minimum=1, maximum=3, value=1, step=1,
                                       label="Number of top morphs to include")
                with gr.Column():
                    comb_text = gr.Markdown("")
                    combination = gr.Gallery()
                    images = gr.Button("Find images", variant="primary")
                    images.click(fn=explore, inputs=[num_morphs], outputs=[combination, comb_text])
                    combination_close = gr.Button("Close")
            with gr.Tab("From morphs"):
                with gr.Column():
                    morph1 = gr.Dropdown(choices=selection, label="Morph 1")
                    morph2 = gr.Dropdown(choices=selection, label="Morph 2")
                    morph3 = gr.Dropdown(choices=selection, label="Morph 3")
                    man_comb_text = gr.Markdown("")
                    man_combination = gr.Gallery()
                    morphs_submit = gr.Button("Find images", variant="primary")
                    morphs_submit.click(fn=man_explore, inputs=[morph1, morph2, morph3],
                                        outputs=[man_combination, man_comb_text])
                    man_combination_close = gr.Button("Close")
    gr.Markdown("## Donate to the project")
    gr.Markdown("If you found this model useful, please consider "
                "[donating to the project](https://ko-fi.com/ethanporcaro)."
                " This will let me pay for faster servers with more computing power and allow me to add more features"
                " to the model. ")
    demo.launch()
