import streamlit as st
import requests
import json
import time
import mixpanel
from mixpanel import Mixpanel
from dotenv import load_dotenv
import os
import pandas as pd
import random
from google.cloud import firestore

# Load environment variables from .env file
load_dotenv()

api_token = os.getenv("API_TOKEN")
mp = Mixpanel(api_token)

# Authenticate to Firestore with the JSON account key.
# db = firestore.Client.from_service_account_info()

# Function to make API request
def make_api_request(prompt):
    url = 'http://api.repelloai.com/repello'
    headers = {'Content-Type': 'application/json'}
    input = {"input" : prompt}
    json_string = json.dumps(input, indent=2)
    data = {
        "data" : json_string
    }
     # Record the start time
    start_time = time.time()
    
    response = requests.post(url, json=data, headers=headers)

    # Calculate the time taken
    end_time = time.time()
    time_taken = end_time - start_time

    return response.json(), time_taken

# Function to create a table for the result categories
def display_result_table(results):
    # Create a table with three columns
    table_data = []
    for model_result in results:
        try:
            threats = model_result.get("threats", {})
            probabilities = model_result.get("probabilities", {})
        except AttributeError:
            st.error("Error retrieving threats and scores.")
            continue
        
        if isinstance(threats, dict) and isinstance(probabilities, dict):
            for threat, probability in probabilities.items():
                emoji_flag = "🚨" if threats.get(threat, False) else "👌"
                true_or_false = str(threats.get(threat, False))
                table_data.append({"Threat": threat, "Detected?": true_or_false, "Probability": probability, "Verdict": emoji_flag})

    # Display the table
    if table_data:
        st.table(table_data)
    else:
        st.text("No results to display.")

# Function to get 4 random prompts from the CSV file
def get_random_prompts():
    csv_file_path = "bad_prompts.csv"
    df = pd.read_csv(csv_file_path)
    random_prompts = df.sample(4)["text"].tolist()
    return random_prompts

# Streamlit app layout
def main():
    #Track the event 'Page View'
    mp.track('Page View', event_name='New Visitor')
    # Set page layout
    st.set_page_config(layout="wide")

    # Initialize session state
    if 'response' not in st.session_state:
        st.session_state.response = None
    if 'selected_prompt' not in st.session_state:
        st.session_state.selected_prompt = ""
    if 'button_texts' not in st.session_state:
        st.session_state.button_texts = []
    if 'hasSent' not in st.session_state:
        st.session_state.prev_response = 0

    # Big, bold heading with magical wand emoji
    st.title("Repello 🪄 Playground")

    # Input box for user prompts
    prompt = st.text_area("Enter your prompt:", value=st.session_state.selected_prompt)
    
    if st.button("Send"):
        if prompt:
            response, time_taken = make_api_request(prompt)
            # Example: Track a custom event 'Button Click'
            mp.track('Button Click', event_name='Api call')
            st.session_state.response = response
            st.session_state.time_taken = time_taken/10
            st.session_state.hasSent = 1

    # Display result table or JSON response below input box
    st.header("Results:")
    if st.session_state.response is not None:
        results = st.session_state.response.get("responseData", {}).get("results", [])
        if results:
            display_result_table(results)

            # Display time taken for the response
            st.subheader("Time Taken for Response ⏱️")
            st.write(f"The response took {st.session_state.time_taken:.4f} seconds.")

            # Button to open Google Form
            st.text("To report an issue write to : naman@strello.co")
            if st.session_state.hasSent: 
                # db.collection("prompts").add({"prompt": st.session_state.selected_prompt})
                st.session_state.hasSent = 0

        else:
            st.text("The detection results of your prompt will appear here.")
    else:
        st.text("The detection results of your prompt will appear here.")

    # Left column with buttons
    st.sidebar.title("Horcrux Prompts 🚫")
    st.sidebar.write("**Try out these perilous prompts which have previously created havoc for LLMs and see if our spell works!**")
    
    if len(st.session_state.button_texts)==0:
        st.session_state.button_texts = get_random_prompts()

    # Button to refresh prompts
    if st.sidebar.button("Refresh Prompts 🔄"):
        # Clear existing button_texts
        st.session_state.button_texts = []
        # Get new random prompts
        st.session_state.button_texts = get_random_prompts()

    for i, text in enumerate(st.session_state.button_texts, start=1):
        if st.sidebar.button(text, key=f"button_{i}", on_click=lambda t=text: st.session_state.update(selected_prompt=t.strip())):
            st.session_state.selected_prompt = text.strip()

if __name__ == "__main__":
    main()