import gradio as gr
import requests
import json
from send_email_request import send_email


def request_api_key_form(req_email, req_name, req_organization, req_meddra_license, req_agree_terms, req_save_data):
    # Check if all required fields are filled and the conditions are met
    if not req_email or not req_name or not req_organization:
        return "**Please fill in all required fields.**"
    if "@" not in req_email or "." not in req_email:
        return "**Please enter a valid email.**"
    if not req_meddra_license:
        return "**You need to have a valid MedDRA license.**"
    if not req_agree_terms:
        return "**You need to agree to Safeterm terms of use.**"
    if not req_save_data:
        return "**You need to agree to share data for training and communication purposes.**"

    subject = "API Key Request for Safeterm"

    send_email(subject, req_email, req_name, req_organization, req_meddra_license, req_agree_terms, req_save_data)

    return "**Your request for API key has been submitted successfully! We will send it to you via mail soon!**"


def encode_caller(apikey, reported_terms_encoder, validation_request, single_term):
    if not apikey:
        return "Please enter a valid API key!"

    url = "https://safeterm.proddis.com/meddra_encode"
    reported_terms_list = [term.strip() for term in reported_terms_encoder.split(",")]

    # Convert string values to boolean
    validation_request_bool = validation_request == "True"
    single_term_bool = single_term == "True"

    payload = json.dumps({
        "reported_terms": reported_terms_list,
        "nmax": 1,
        "version": float("26.0"),
        "validation_request": validation_request_bool,
        "single_term": single_term_bool
    })

    headers = {
        'Content-Type': 'application/json',
        'Authorization': f'Bearer {apikey}'
    }

    response = requests.post(url, headers=headers, data=payload)
    data = response.json()

    if "detail" in data:
        return data["detail"]

    results = []

    for encoded_term in data.get("encoded_terms", []):
        result = f"Reported Term: {encoded_term['term']}\n"

        # Check if 'encoded_term' contains a list of dictionaries (valid_request = False)
        if isinstance(encoded_term['encoded_term'][0], dict):
            for sub_term_entry in encoded_term['encoded_term']:
                for llt_id, llt_term in zip(sub_term_entry['llt_id'], sub_term_entry['llt_term']):
                    result += f"LLT ID: {llt_id}\nLLT Term: {llt_term}\n"

                pt_id = sub_term_entry['pt_id'][0]
                pt_term = sub_term_entry['pt_term'][0]
                result += f"PT ID: {pt_id}\nPT Term: {pt_term}\n"
                result += "---\n"

        # Case for valid_request = True (list of lists)
        else:
            for sub_list in encoded_term['encoded_term']:
                for sub_term_entry in sub_list:
                    if 'llt_id' in sub_term_entry and 'llt_term' in sub_term_entry:
                        for llt_id, llt_term in zip(sub_term_entry['llt_id'], sub_term_entry['llt_term']):
                            result += f"LLT ID: {llt_id}\nLLT Term: {llt_term}\n"

                        pt_id = sub_term_entry['pt_id'][0]
                        pt_term = sub_term_entry['pt_term'][0]
                        result += f"PT ID: {pt_id}\nPT Term: {pt_term}\n"
                        result += "---\n"

                    # Handle other entries like "is_validated" and "report"
                    else:
                        for key, value in sub_term_entry.items():
                            result += f"{key}: {value}\n"
                        result += "---\n"

        result += f"Status: {encoded_term['status']}\n\n"
        results.append(result)

    # Add the API messages at the end.
    api_message = data.get("messages", "No API message available")
    api_message = "OK" if api_message is None else api_message
    results.append(f"API Message: {api_message}")

    return "\n".join(results)


def validate_caller(val_apikey, val_reported_terms, val_llt_terms):
    if not val_apikey:
        return "Please enter a valid API key!"
    url = "https://safeterm.proddis.com/meddra_validate"

    # Convert comma-separated strings into lists
    val_reported_terms_list = [term.strip() for term in val_reported_terms.split(",")]
    val_llt_terms_list = [term.strip() for term in val_llt_terms.split(",")]

    payload = json.dumps({
        "reported_terms": val_reported_terms_list,
        "llt_terms": val_llt_terms_list,
        "nmax": 1,
        "version": float("26.0")
    })
    headers = {
        'Content-Type': 'application/json',
        'Authorization': f'Bearer {val_apikey}'
    }

    response = requests.request("POST", url, headers=headers, data=payload, timeout=60)

    data = response.json()

    # Check if the response contains an error detail
    if "detail" in data:
        return data["detail"]

    results = []
    for validation_item in data["responses"]:
        report = validation_item.get("report", "No report")

        if validation_item["best_dict_term"]:
            llt_id = validation_item["best_dict_term"][0]["llt_id"][0]
            llt_term = validation_item["best_dict_term"][0]["llt_term"][0]
            pt_id = validation_item["best_dict_term"][0]["pt_id"][0]
            pt_term = validation_item["best_dict_term"][0]["pt_term"][0]

            result = f"Report: {report}\n"
            result += f"LLT ID: {llt_id}\n"
            result += f"LLT Term: {llt_term}\n"
            result += f"PT ID: {pt_id}\n"
            result += f"PT Term: {pt_term}\n\n"
        else:
            result = f"Report: {report}\nNo matching terms found.\n\n"

        results.append(result)

        # Add the API messages at the end.
        api_message = data.get("messages", "No API message available")
        api_message = "OK" if api_message is None else api_message
        results.append(f"API key status: {api_message}")

    return "\n".join(results)


def upgrade_caller(api_key, upgrader_list_of_terms, version_upgrade):
    if not api_key:
        return "Please enter a valid API key!"
    url = "https://safeterm.proddis.com/meddra_version_upgrade"

    cleaned_terms = [term.strip() for term in upgrader_list_of_terms.split(",")]

    payload = json.dumps({
        "list_of_terms": cleaned_terms,
        "nmax": 1,
        "version": float(version_upgrade),
        "verbose": True
    })
    headers = {
        'Content-Type': 'application/json',
        'Authorization': f'Bearer {api_key}'
    }
    response = requests.request("POST", url, headers=headers, data=payload)

    data = response.json()

    # Check if the response contains an error detail
    if "detail" in data:
        return data["detail"]

    output = []
    results = data.get("result", [])
    for i, result in enumerate(results):
        input_term = cleaned_terms[i]  # Assuming the order in 'results' and 'cleaned_terms' is the same

        if result['change_status'] is None or result['output_term'] is None:
            status_message = result.get('status', 'Unknown Status')
            output.append(f"Input Term: {input_term}\nStatus: {status_message}\n")
        else:
            status = "Unchanged" if not result['change_status'] else "Changed"
            output_term = result['output_term']
            output.append(f"Input Term: {input_term}\nStatus: {status}\nOutput Term: {output_term}\n")

    # Get the API message, set to "OK" if it's None
    api_message = data.get("messages", "No API message available")
    api_message = "OK" if api_message is None else api_message

    output.append(f"API status: {api_message}")

    return "\n".join(output).strip()  # Removing any trailing whitespace


def llt_current_caller(api_key, currency_list_of_terms, version_check):
    if not api_key:
        return "Please enter a valid API key!"

    url = "https://safeterm.proddis.com/meddra_llt_is_current"
    cleaned_terms_currency = [term.strip() for term in currency_list_of_terms.split(",")]

    payload = json.dumps({
        "list_of_terms": cleaned_terms_currency,
        "version": float(version_check)
    })

    headers = {
        'Content-Type': 'application/json',
        'Authorization': f'Bearer {api_key}'
    }

    response = requests.request("POST", url, headers=headers, data=payload)
    response_data = response.json()

    # Check if 'is_current_flag_results' key exists in the API response
    if "is_current_flag_results" not in response_data:
        # If 'detail' key exists in response_data, show the error detail, else show a generic error
        return response_data.get("detail", "An error occurred.")

    beautified_output = []
    for term, flag_data in zip(cleaned_terms_currency, response_data["is_current_flag_results"]):
        flag = flag_data.get('is_current_flag', None)
        status = flag_data.get('status', '')

        if isinstance(flag, bool):
            status_text = "Is Current" if flag else "Is not Current"
            beautified_output.append(f"{term}: {status_text}")

        else:
            beautified_output.append(f"Error {term}: {status}")

    # Get the API message, set to "OK" if it's None
    api_message = response_data.get("messages", None)
    api_message = "OK" if api_message is None else api_message
    beautified_output.append(f"API status: {api_message}")

    return "\n".join(beautified_output)


with gr.Blocks(css=".gradio-container {background-color: lightgray}") as demo:
    # gr.Markdown("Safeterm is an automated AI system that extracts medical terms from patient narratives and
    # standardize these terms according to the medical dictionary for regulatory applications (MedDRA). ")
    gr.Markdown("### Safeterm: Translate Medical Narratives into Standardized Dictionaries")
    with gr.Row():
        with gr.Column():
            gr.HTML(
                """<p>Safeterm is an automated AI system that extracts medical terms from patient narratives and 
                standardize these terms according to the medical dictionary for regulatory applications (MedDRA). 
                </p>""")
        with gr.Column():
            universal_api_key_input = gr.Textbox(label="Paste your API Key", placeholder="Enter your API Key...",
                                                 lines=1)
    with gr.Row():
        with gr.Tab("Safeterm Encode"):
            gr.Markdown("### Safeterm Encode")

            # Inputs
            encode_reported_terms = gr.Dropdown(
                ["Headache, Allergic to CAT scan", "Myocardial infarction in the fall of 2000",
                 "Nurse miscalculated the dose and the patient received 40 mg instead of 20 mg of his medication. He "
                 "experienced severe hypotension and heart attack."],
                label="Medical Narratives (comma-separated, max 5)",
                info="Enter your reported medical narratives (1 or more comma-separated statements, example: "
                     "Headache, Allergic to CAT scan) here or choose from the presets.",
                allow_custom_value=True)

            encode_version = gr.Textbox(label="MedDRA Version", value="26.0")

            # New input boxes
            validation_request = gr.Radio(label="Validation Request", choices=["True", "False"])
            single_term = gr.Radio(label="Single Term", choices=["True", "False"])

            # Output
            api_response_encode = gr.Textbox(label="Standardized Medical Dictionary Outputs")

            # Button to trigger API call
            submit_button = gr.Button("Encode into MedDRA")

            # Binding API call function to button (Note: Assuming `universal_api_key_input` was defined somewhere
            # before this block)
            submit_button.click(encode_caller,
                                inputs=[universal_api_key_input, encode_reported_terms, validation_request,
                                        single_term],
                                outputs=api_response_encode)

        with gr.Tab("Safeterm Validate"):
            gr.Markdown('### Safeterm Validate')
            gr.Markdown(
                'Checks an existing MedDRA encoding. Compares reported terms to MedDRA Lower Level Terms (LLT). '
                'Provides an alternative LLT in case of discrepancy.')
            # Validator Inputs
            reported_terms_validate = gr.Dropdown(["Pain in the head, Feeling nauseous"]
                                                  , label="Reported Terms (comma-separated, max 5)"
                                                  ,
                                                  info="Enter your reported terms here (example: pain in the "
                                                       "head, Feeling nauseous) or choose from the dropdown preset",
                                                  allow_custom_value=True)
            llt_terms_validate = gr.Dropdown(["Headache, Vomiting"]
                                             , label="LLT terms (comma-separated, max 5)"
                                             ,
                                             info="Enter the current LLTs here (example: Headache, Vomitting), "
                                                  "or choose"
                                                  "from the dropdown presets",
                                             allow_custom_value=True)
            version_validate = gr.Textbox(label="MedDRA Version", value="26.0")

            # Output
            api_response_validate = gr.Textbox(label="Validation Report")

            # Button to trigger API call and Binding API call function to button for Validator
            submit_button_validate = gr.Button("Validate Existing MedDRA Encoding")
            submit_button_validate.click(validate_caller,
                                         inputs=[universal_api_key_input, reported_terms_validate, llt_terms_validate],
                                         outputs=api_response_validate)
        with gr.Tab("Safeterm Version Upgrade"):
            gr.Markdown("### Safeterm Version Upgrade")
            gr.Markdown('Upgrade terms to the most up-to-date LLTs in a MedDRA version.')
            # Version Upgrade Inputs
            list_of_terms_upgrade = gr.Dropdown(
                ["Pain in the head", "Injection site joint inflammation", "Bone chip removal"]
                , label="LLTs (comma-separated, max 5)"
                , info="Enter your LLTs here (example: Injection site joint inflammation) "
                       "or choose from the dropdown preset",
                allow_custom_value=True)

            version_upgrade = gr.Dropdown(label="To MedDRA Version", choices=["20.0", "26.0"])

            # Output
            api_response_upgrade = gr.Textbox(label="Version Upgrade Response")

            # Button to trigger API call and Binding API call function to button for Version Upgrade
            submit_button_upgrade = gr.Button("Upgrade to Selected MedDRA Version")
            submit_button_upgrade.click(upgrade_caller,
                                        inputs=[universal_api_key_input, list_of_terms_upgrade,
                                                version_upgrade], outputs=api_response_upgrade)

        with gr.Tab("Safeterm Current Check"):  # Currency Checker section
            gr.Markdown("### Safeterm Current Version Check")
            gr.Markdown('Checks if LLTs are current or not in a particular MedDRA version')
            # No Change Version Check Inputs
            list_of_terms_validity = gr.Dropdown(["Anemia iron deficiency", "COVID-19"]
                                                 , label="LLTs (comma-separated, max 5)"
                                                 , info="Enter your LLTs here, e.g.: Injection site "
                                                        "joint inflammation or choose from the dropdown preset",
                                                 allow_custom_value=True)
            version_check = gr.Dropdown(label="To MedDRA Version", choices=["20.0", "26.0"])

            # Output for No Change Version Check
            api_response_current = gr.Textbox(label="Current Check Response")

            # Button to trigger API call and Binding API call function to button for No Change Check
            submit_button_currency = gr.Button("Check if LLT is current or not")
            submit_button_currency.click(llt_current_caller,
                                         inputs=[universal_api_key_input, list_of_terms_validity, version_check],
                                         outputs=api_response_current)

        with gr.Tab("Request an API key"):
            # gr.Markdown("### Safeterm Settings")
            gr.Markdown("### Request an API key")
            free_demo = gr.Markdown(
                "Submit your request for a free demo of our API (Expires in 30 days. 50 terms limit)")
            # gr.Markdown("### MedDRA Dictionary")
            version = gr.Markdown("Safeterm Model v-082023")
            language = gr.Markdown("Language: English")  # Dropdown for language

            gr.Markdown("### Contact information")
            email = gr.Textbox(label="Email", placeholder="Enter your professional email address...")
            name = gr.Textbox(label="First and Last Name", placeholder="Enter your full name...")
            organization = gr.Textbox(label="Organization", placeholder="Enter your organization name and details...")

            gr.Markdown("### Terms of use")
            # MedDRA_license = gr.Checkbox(label="I confirm that my organization has a valid MedDRA license.")
            with gr.Row():
                MedDRA_license = gr.Checkbox(label="I confirm that my organization has a valid MedDRA license: ")
                gr.HTML("""<a href=https://www.meddra.org/subscription/process>[link]</a>""")
            with gr.Row():
                agree_terms = gr.Checkbox(label="I agree to Safeterm terms of use: ", min_width=1, scale=1)
                gr.HTML("""<a href=https://www.proddis.com/pdf/Proddis_Terms_of_Use.pdf>[link] </a>""")

            with gr.Row():
                save_data = gr.Checkbox(
                    label="I consent to the storage of my personal data for training and communication purposes.")
                gr.HTML(""" """)

            feedback_textbox = gr.Markdown(label="Feedback")

            # Button for API key request and Binding function to button
            api_key_button = gr.Button("Submit request for API key")
            api_key_button.click(
                request_api_key_form,
                inputs=[email, name, organization, MedDRA_license, agree_terms, save_data],
                outputs=feedback_textbox
            )
    with gr.Row():
        gr.Markdown(
            "Safeterm API is distributed by ClinBAY Limited. \t For any enquiry, feel free to contact us at "
            "info@clinbay.com")

demo.launch()
