import copy
import unicodedata

YALE_TO_HANGUL_INITIAL_CONSONANTS = {
    'k': '\u1100',
    'kk': '\u1101',
    'n': '\u1102',
    'nn': '\u1114',
    't': '\u1103',
    'tt': '\u1104',
    'l': '\u1105',
    'm': '\u1106',
    'p': '\u1107',
    'pp': '\u1108',
    's': '\u1109',
    'ss': '\u110a',
    'G': '\u110B',
    'GG': '\u1147',
    'c': '\u110C',
    'cc': '\u110D',
    'ch': '\u110e',
    'kh': '\u110f',
    'th': '\u1110',
    'ph': '\u1111',
    'h': '\u1112',

    'pk': '\u111e',
    'pt': '\u1120',
    'ps': '\u1121',
    'psk': '\u1122',
    'pst': '\u1123',
    'pc': '\u1127',
    'pth': '\u1129',
    'W': '\u112b',
    'sk': '\u112d',
    'sn': '\u112e',
    'st': '\u112f',
    'sp': '\u1132',
    'sc': '\u1136',
    'sh': '\u113b',
    'z': '\u1140',
    'hh': '\u1158',
    'q': '\u1159',

    'ng': '\u114c',

    '': '\u115f',
}

YALE_TO_HANGUL_FINAL_CONSONANTS = {
    'k': '\u11a8',
    'ks': '\u11aa',
    'n': '\u11ab',
    't': '\u11ae',
    'l': '\u11af',
    'lk': '\u11b0',
    'lm': '\u11b1',
    'lp': '\u11b2',
    'ls': '\u11b3',
    'm': '\u11b7',
    'p': '\u11b8',
    'ps': '\u11b9',
    's': '\u11ba',
    'G': '\u11bc',
    'nt': '\u11c6',
    'ns': '\u11c7',
    'nz': '\u11c8',
    'lks': '\u11cc',
    'lz': '\u11d7',
    'lq': '\u11d9',
    'mp': '\u11dc',
    'ms': '\u11dd',
    'mz': '\u11df',
    'M': '\u11e2',
    'W': '\u11e6',
    'z': '\u11eb',
    'ng': '\u11f0',
    'q': '\u11f9',
    'ngs': '\u11f1',
    '': ''
}

YALE_TO_HANGUL_VOWELS = {
    'a': '\u1161',
    'ay': '\u1162',
    'ya': '\u1163',
    'yay': '\u1164',
    'e': '\u1165',
    'ey': '\u1166',
    'ye': '\u1167',
    'yey': '\u1168',
    'wo': '\u1169',
    'wa': '\u116a',
    'way': '\u116b',
    'woy': '\u116c',
    'yo': '\u116d',
    'wu': '\u116e',
    'we': '\u116f',
    'wey': '\u1170',
    'wuy': '\u1171',
    'yu': '\u1172',
    'u': '\u1173',
    'uy': '\u1174',
    'i': '\u1175',

    'o': '\u119e',
    'oy': '\u11a1',
    'yoy': '\u1188',
    'yuy': '\u1194',
    'ywe': '\u1191',
    'ywey': '\u1192',
    'ywa': '\u1184',
    'yway': '\u1185'
}

UNICODE_COMPATIBILITY_FORMS = {
    'ᄀ': 'ㄱ',
    'ᄁ': 'ㄲ',
    'ᆪ': 'ㄳ',
    'ᄂ': 'ㄴ',
    'ᆬ': 'ㄵ',
    'ᆭ': 'ㄶ',
    'ᄃ': 'ㄷ',
    'ᄄ': 'ㄸ',
    'ᄅ': 'ㄹ',
    'ᆰ': 'ㄺ',
    'ᆱ': 'ㄻ',
    'ᆲ': 'ㄼ',
    'ᆳ': 'ㄽ',
    'ᆴ': 'ㄾ',
    'ᆵ': 'ㄿ',
    'ᄚ': 'ㅀ',
    'ᄆ': 'ㅁ',
    'ᄇ': 'ㅂ',
    'ᄈ': 'ㅃ',
    'ᄡ': 'ㅄ',
    'ᄉ': 'ㅅ',
    'ᄊ': 'ㅆ',
    'ᄋ': 'ㅇ',
    'ᄌ': 'ㅈ',
    'ᄍ': 'ㅉ',
    'ᄎ': 'ㅊ',
    'ᄏ': 'ㅋ',
    'ᄐ': 'ㅌ',
    'ᄑ': 'ㅍ',
    'ᄒ': 'ㅎ',
    'ᅡ': 'ㅏ',
    'ᅢ': 'ㅐ',
    'ᅣ': 'ㅑ',
    'ᅤ': 'ㅒ',
    'ᅥ': 'ㅓ',
    'ᅦ': 'ㅔ',
    'ᅧ': 'ㅕ',
    'ᅨ': 'ㅖ',
    'ᅩ': 'ㅗ',
    'ᅪ': 'ㅘ',
    'ᅫ': 'ㅙ',
    'ᅬ': 'ㅚ',
    'ᅭ': 'ㅛ',
    'ᅮ': 'ㅜ',
    'ᅯ': 'ㅝ',
    'ᅰ': 'ㅞ',
    'ᅱ': 'ㅟ',
    'ᅲ': 'ㅠ',
    'ᅳ': 'ㅡ',
    'ᅴ': 'ㅢ',
    'ᅵ': 'ㅣ',
    'ᄔ': 'ㅥ',
    'ᄕ': 'ㅦ',
    'ᇇ': 'ㅧ',
    'ᇈ': 'ㅨ',
    'ᇌ': 'ㅩ',
    'ᇎ': 'ㅪ',
    'ᇓ': 'ㅫ',
    'ᇗ': 'ㅬ',
    'ᇙ': 'ㅭ',
    'ᄜ': 'ㅮ',
    'ᇝ': 'ㅯ',
    'ᇟ': 'ㅰ',
    'ᄝ': 'ㅱ',
    'ᄞ': 'ㅲ',
    'ᄠ': 'ㅳ',
    'ᄢ': 'ㅴ',
    'ᄣ': 'ㅵ',
    'ᄧ': 'ㅶ',
    'ᄩ': 'ㅷ',
    'ᄫ': 'ㅸ',
    'ᄬ': 'ㅹ',
    'ᄭ': 'ㅺ',
    'ᄮ': 'ㅻ',
    'ᄯ': 'ㅼ',
    'ᄲ': 'ㅽ',
    'ᄶ': 'ㅾ',
    'ᅀ': 'ㅿ',
    'ᅇ': 'ㆀ',
    'ᅌ': 'ㆁ',
    'ᇱ': 'ㆂ',
    'ᇲ': 'ㆃ',
    'ᅗ': 'ㆄ',
    'ᅘ': 'ㆅ',
    'ᅙ': 'ㆆ',
    'ᆄ': 'ㆇ',
    'ᆅ': 'ㆈ',
    'ᆈ': 'ㆉ',
    'ᆑ': 'ㆊ',
    'ᆒ': 'ㆋ',
    'ᆔ': 'ㆌ',
    'ᆞ': 'ㆍ',
    'ᆡ': 'ㆎ',
}


def convert_yale_to_hangul(yale):
    syllables = yale.split('.')

    result = ""
    for syllable in syllables:
        out_syll = ""
        tone_mark = ""

        orig_syllable = copy.copy(syllable)

        if any(syllable.endswith(t) for t in ['L', "H", "R"]):
            tone_mark = {
                'L': '',
                'H': '\u302e',
                'R': '\u302f'
            }[syllable[-1]]
            syllable = syllable[:-1]

        initial_exists = False
        for n in range(4, -1, -1):
            if syllable[:n] in YALE_TO_HANGUL_INITIAL_CONSONANTS:
                out_syll += YALE_TO_HANGUL_INITIAL_CONSONANTS[syllable[:n]]
                syllable = syllable[n:]
                initial_exists = (n > 0)
                break

        vowel_exists = False
        for n in range(4, 0, -1):
            if syllable[:n] in YALE_TO_HANGUL_VOWELS:
                out_syll += YALE_TO_HANGUL_VOWELS[syllable[:n]]
                syllable = syllable[n:]
                vowel_exists = True
                break

        for n in range(4, 0, -1):
            if syllable[:n] in YALE_TO_HANGUL_FINAL_CONSONANTS:
                out_syll += YALE_TO_HANGUL_FINAL_CONSONANTS[syllable[:n]]
                syllable = syllable[n:]
                break

        out_syll += tone_mark

        if initial_exists and not vowel_exists and tone_mark == "":
            if out_syll in UNICODE_COMPATIBILITY_FORMS:
                out_syll = UNICODE_COMPATIBILITY_FORMS[out_syll]

        if not initial_exists and vowel_exists and tone_mark == "":
            if out_syll[1:] in UNICODE_COMPATIBILITY_FORMS:
                out_syll = UNICODE_COMPATIBILITY_FORMS[out_syll[1:]]

        if len(syllable) > 0:
            # Failed to convert
            out_syll = orig_syllable

        result += out_syll

    return result
