import warnings

from pytorch_lightning.callbacks import ModelCheckpoint

warnings.filterwarnings(
    action="ignore",
    message=".*An output with one or more elements.*",
)
warnings.filterwarnings(
    action="ignore",
    message=".*mask with dtype torch.uint8.*",
)
warnings.filterwarnings(
    action="ignore",
    message=".*indexing with dtype torch.uint8.*",
)

from typing import List
import argparse

import numpy as np
import pytorch_lightning as L
import torch.utils.data
from pytorch_lightning.loggers import TensorBoardLogger

from model import exkp, compute_loss
from synthetic_dataset import load_dataset


class CenterNet(L.LightningModule):
    def __init__(self):
        super().__init__()
        self.model = exkp(
            n=5,
            nstack=4,
            dims=[256, 256, 384, 384, 384, 512],
            modules=[2, 2, 2, 2, 2, 4],
            num_classes=4
        )

    def forward(self, x):
        return self.model(x)

    def validation_step(self, batch, batch_idx):
        outputs = self(batch['image'])
        loss, loss_dict = compute_loss(outputs, batch, return_dict=True)
        self.log("val_loss", loss.mean(), sync_dist=True)
        for key, value in loss_dict.items():
            self.log("val_" + key, value.mean(), sync_dist=True)

    def training_step(self, batch, batch_idx):
        outputs = self(batch['image'])
        loss, loss_dict = compute_loss(outputs, batch, return_dict=True)
        self.log("train_loss", loss.mean(), sync_dist=True)
        for key, value in loss_dict.items():
            self.log("train_" + key, value.mean(), sync_dist=True)
        return loss.mean()

    def configure_optimizers(self):
        optimizer = torch.optim.Adam(self.parameters(), 2.5e-4)
        scheduler = torch.optim.lr_scheduler.MultiStepLR(optimizer, [45, 60], gamma=0.1)
        return [optimizer], [scheduler]


def collate(batch: List[dict]):
    out = {k: [] for k in batch[0].keys()}
    for item in batch:
        for key, value in item.items():
            out[key].append(value)
    result = {}
    for key, value in out.items():
        if any(isinstance(v, np.ndarray) for v in value):
            valid = torch.tensor([isinstance(v, np.ndarray) for v in value])
            result[f"{key}_valid"] = valid
            zero_array = np.zeros_like([v for v in value if isinstance(v, np.ndarray)][0])
            filled_value = [v if isinstance(v, np.ndarray) else zero_array for v in value]
            value = torch.as_tensor(np.stack(filled_value))

        result[key] = value
    return result


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--resume', default=None)
    parser.add_argument('--ckpt_path', required=True, type=str)
    args = parser.parse_args()

    train_dataset, test_dataset = load_dataset(train_size=10**6)

    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=2,
        shuffle=True, num_workers=2, pin_memory=True,
        drop_last=True, collate_fn=collate)
    val_loader = torch.utils.data.DataLoader(
        test_dataset, batch_size=2,
        shuffle=False, num_workers=2, pin_memory=True,
        drop_last=True, collate_fn=collate)

    logger = TensorBoardLogger("tb_logs", name="centernet")

    checkpoint_callbacks = [
        ModelCheckpoint(
            dirpath=args.ckpt_path,
            save_top_k=5,
            monitor="val_loss",
            every_n_epochs=1,
            save_last=True
        ),
    ]
    trainer = L.Trainer(
        accelerator="gpu",
        logger=logger,
        max_epochs=-1,
        callbacks=checkpoint_callbacks,
        val_check_interval=200,
        check_val_every_n_epoch=None,
        accumulate_grad_batches=4
    )

    centernet = CenterNet()
    trainer.fit(centernet, train_loader, val_loader,
                ckpt_path=args.resume)


if __name__ == "__main__":
    main()
