import json

import cv2
from torch.utils.data import Dataset
import random
import math
import albumentations
from sklearn.model_selection import train_test_split
import pathlib
import numpy as np
from scipy.ndimage import map_coordinates
from scipy.ndimage import gaussian_filter

from load_book import process_page
from utils.image import gaussian_radius, draw_gaussian


def resize(image, dsize, fx=None, fy=None, interpolation=cv2.INTER_AREA):
    image, bboxes = image
    orig_height, orig_width = image.shape[0], image.shape[1]
    image = cv2.resize(image, dsize=dsize, fx=fx, fy=fy, interpolation=interpolation)
    H, W = image.shape[0], image.shape[1]
    bboxes = bboxes * np.array([W / orig_width, H / orig_height])
    return image, bboxes


def crop(image, tlx, tly, brx, bry):
    image, bboxes = image
    image = image[tly:bry, tlx:brx]
    bboxes = bboxes - np.array([tlx, tly])
    return image, bboxes


def make_line(file_list, do_stretch, remove_margin, rand):
    images = []
    syllables = []

    for p in file_list:
        img = cv2.imread(str(p), cv2.IMREAD_UNCHANGED)  # [H, W, rgba]
        img = cv2.resize(img, dsize=None, fx=0.5, fy=0.5, interpolation=cv2.INTER_AREA)
        alpha = img[:, :, 3:] / 255.
        img = np.uint8(img[:, :, :-1] * alpha + 255 * (1 - alpha))  # white background
        image_grey = 255 - cv2.cvtColor(img, cv2.COLOR_RGB2GRAY)
        tx, ty, w, h = cv2.boundingRect(image_grey)
        img = (img, np.array([[[tx, ty], [tx + w, ty + h]]]))

        syllables.append(p.stem)

        orig_height, orig_width = img[0].shape[0], img[0].shape[1]

        # remove top & bottom margins
        img = crop(img, 0, max(0, ty - 4), img[0].shape[1], min(ty + h + 4, img[0].shape[0]))

        if do_stretch:
            img = resize(img, dsize=None, fx=1.0, fy=rand.uniform(0.6, 1.0), interpolation=cv2.INTER_AREA)
        else:
            img = resize(img, dsize=(orig_width, orig_height), interpolation=cv2.INTER_AREA)

        if remove_margin:
            # remove left margin
            img = crop(img, tx, 0, min(tx + w + 2, img[0].shape[1]), img[0].shape[0])
            img = resize(img, dsize=(orig_width, orig_height), interpolation=cv2.INTER_AREA)

        images.append(img)

    img = vstack(images)

    return img


def vstack(images):
    offset = 0
    all_coords = []
    for img, coords in images:
        all_coords.append(coords + np.array([0, offset]))
        offset += img.shape[0]
    all_coords = np.concatenate(all_coords)
    stacked_image = np.vstack([img for img, _ in images])
    return stacked_image, all_coords


def add_border(img, top, bottom, left, right, value):
    return cv2.copyMakeBorder(
        img[0], top, bottom, left, right, cv2.BORDER_CONSTANT, None, value
    ), img[1] + np.array([left, top])


class SyllableDataset(Dataset):
    def __init__(self, file_list, hanja_file_list,
                 real_annotations,
                 do_augment=None, length=100000):
        self.files_no_final = [
            [f for f in font if f.stem[-1] not in 'aeiouy']
            for font in file_list
        ]
        self.files_with_final = [
            [f for f in font if f.stem[-1] in 'aeiouy']
            for font in file_list
        ]
        self.hanja_file_list = hanja_file_list
        self.real_annotations = real_annotations
        self.do_augment = do_augment
        self.length = length

    def __len__(self):
        return self.length

    def __getitem__(self, idx):
        rand = random.Random(idx)
        max_objs = 9 * 16 * 4

        if rand.randint(0, 1) == 0:
            record = rand.choice(self.real_annotations)

            file_url = record['data']['url']
            annotations = record['annotations'][0]['result']

            filename = '/'.join(file_url.split('/')[-2:])
            image, orig_image_bbox, orig_size = process_page(filename, thresholding=False)

            bboxes = np.array([
                [[rect['value']['x'], rect['value']['y']],
                 [rect['value']['x'] + rect['value']['width'], rect['value']['y'] + rect['value']['height']]]
                for rect in annotations
            ])
            bboxes = bboxes / 100. * np.array(orig_size) - np.array(orig_image_bbox[0])

            labels = np.array([
                ['kor_main', 'hanja_main', 'kor_anno', 'hanja_anno'].index(rect['value']['rectanglelabels'][0])
                for rect in annotations
            ])

            # clip bboxes
            bboxes = bboxes.clip(min=0, max=np.array([image.shape[1], image.shape[0]]))
            bboxes = bboxes[(bboxes[:, 0, 0] < bboxes[:, 1, 0]) & (bboxes[:, 0, 1] < bboxes[:, 1, 1])]

            # Augment image
            if self.do_augment is not None:
                if rand.randint(0, 1) == 1:
                    image, bboxes = self.do_augment(image, bboxes)

            # normalize image
            orig_size = (image.shape[1], image.shape[0])
            image = cv2.resize(image, dsize=(512, 512), interpolation=cv2.INTER_AREA)
            bboxes = bboxes * np.array([512 / orig_size[0], 512 / orig_size[1]])
            image = image.astype(np.float32) / 255. - .5  # to [-.5, +.5] range
            image = image.transpose((2, 0, 1))  # [H, W, C] to [C, H, W]

            hmap = make_heatmap(bboxes, labels, max_objs, num_classes=4)

            # pad bboxes
            bboxes = np.concatenate([bboxes, np.zeros((max_objs - bboxes.shape[0], 2, 2))], axis=0)

            return {
                'image': image,
                'bboxes': bboxes,
                'syllables': None,   # TODO
                'sequence': None,
                **hmap
            }

        else:
            return generate_synthetic_page(
                rand,
                self.files_with_final,
                self.files_no_final,
                self.hanja_file_list,
                self.do_augment,
                max_objs
            )


def generate_synthetic_page(rand, files_with_final, files_no_final, hanja_file_list, do_augment, max_objs):
    max_lines = rand.randint(6, 9)
    line_max_len = rand.randint(9, 16)
    L = max_lines * line_max_len * 4
    max_span_len = 50

    add_rule = rand.randint(0, 1) == 1
    do_stretch = rand.randint(0, 1) == 1
    do_remove_margin = rand.randint(0, 1) == 1

    file_list = []
    labels = []

    lines = []
    line_imgs = []
    offset = 0
    page_offset = 0

    is_normal = rand.randint(0, 1) == 1
    while len(lines) < max_lines:
        span_len = rand.randint(1, min(max_span_len, L - offset))

        is_hanja = rand.choices([0, 1], [0.8, 0.2], k=span_len)
        has_final = rand.choices([0, 1], [0.6, 0.4], k=span_len)

        font_idx = rand.randint(0, len(files_with_final) - 1)
        with_finals = files_with_final[font_idx]
        no_finals = files_no_final[font_idx]
        hanjas = rand.choice(hanja_file_list)

        for h, f in zip(is_hanja, has_final):
            if h:
                file_list.append(rand.choice(hanjas))
            else:
                if f:
                    file_list.append(rand.choice(with_finals))
                else:
                    file_list.append(rand.choice(no_finals))

        # 0: normal, hangul
        # 1: normal, hanja
        # 2: annotation, hangul
        # 3: annotation, hanja
        labels.extend(int(not is_normal) * 2 + np.array(is_hanja))

        if is_normal:  # normal span
            # print(f"new normal span {span_len=}")
            while span_len > 0:
                # print(f"main: line {len(lines)}, sec {len(line_imgs)}, {page_offset=}, {span_len=}")
                bit_len = min(span_len, int(line_max_len - page_offset))
                line_imgs.append(make_line(file_list[offset:offset + bit_len],
                                           do_stretch, do_remove_margin, rand))
                offset += bit_len
                page_offset += bit_len
                span_len -= bit_len
                if math.ceil(page_offset) >= line_max_len:
                    page_offset = 0
                    lines.append(vstack(line_imgs))
                    line_imgs = []

        else:  # narrow annotation span
            # print(f"new narrow span {span_len=}")
            while span_len > 0:
                # where in the line the annotation ends
                # print(f"anno: line {len(lines)}, sec {len(line_imgs)}, {page_offset=}, {span_len=}")
                anno_line_max_len = min(line_max_len, page_offset + (span_len + 1) // 2 / 2)

                anno_lines = []

                bit_len = min(span_len, int((anno_line_max_len - page_offset) * 2))
                # print(f"{anno_line_max_len=}, {bit_len=}")
                anno_lines.append(make_line(file_list[offset:offset + bit_len],
                                            do_stretch, do_remove_margin, rand))
                offset += bit_len
                span_len -= bit_len

                bit_len = min(span_len, int((anno_line_max_len - page_offset) * 2))
                # print(f"{anno_line_max_len=}, {bit_len=}")
                if bit_len > 0:
                    anno_lines.append(make_line(file_list[offset:offset + bit_len],
                                                do_stretch, do_remove_margin, rand))
                else:
                    anno_lines.append((np.full_like(anno_lines[0][0], 255), np.zeros((0, 2, 2))))
                offset += bit_len
                span_len -= bit_len

                anno_line = hstack_lines(anno_lines)
                anno_line = resize(anno_line, dsize=None, fx=.45, fy=.5, interpolation=cv2.INTER_AREA)
                anno_line = add_border(anno_line, 0, 0, 128 - anno_line[0].shape[1], 0, [255, 255, 255])
                line_imgs.append(anno_line)

                page_offset = anno_line_max_len

                if math.ceil(page_offset) >= line_max_len:
                    page_offset = 0
                    lines.append(vstack(line_imgs))
                    line_imgs = []

        is_normal = not is_normal

    new_lines = []
    for l, coords in lines:
        tx, ty, w, h = cv2.boundingRect(255 - cv2.cvtColor(l, cv2.COLOR_RGB2GRAY))
        x1, x2 = max(0, tx - 2), min(tx + w + 2, l.shape[1])
        new_lines.append((l[:, x1:x2], coords - np.array([x1, 0])))
    lines = new_lines

    # add indent
    indent = rand.randint(0, 200)
    lines = [add_border(img, indent, 0, 0, 0, [255, 255, 255]) for img in lines]

    if add_rule:
        lines = [add_border(img, 0, 0, 1, 0, 0) for img in lines]

    img = hstack_lines_resize(lines[:max_lines])

    # Add page borders
    img = add_border(img, 1, 1, 1, 1, [255, 255, 255])
    if rand.randint(0, 1) == 1:
        img = add_border(img, 1, 1, 1, 1, 0)
        img = add_border(img, 7, 7, 7, 7, [255, 255, 255])
        img = add_border(img, 10, 10, 10, 10, 0)
        img = add_border(img, 5, 5, 5, 5, [255, 255, 255])

    # Clip bboxes
    img = (img[0], img[1].clip(min=0, max=np.array([img[0].shape[1], img[0].shape[0]])))

    # Augment image
    if do_augment is not None:
        if rand.randint(0, 1) == 1:
            img = do_augment(img[0], img[1])

    image, bboxes = img

    # Normalize image
    orig_size = (image.shape[1], image.shape[0])
    image = cv2.resize(image, dsize=(512, 512), interpolation=cv2.INTER_AREA)
    bboxes = bboxes * np.array([512 / orig_size[0], 512 / orig_size[1]])

    image = image.astype(np.float32) / 255. - .5  # to [-.5, +.5] range
    image = image.transpose((2, 0, 1))  # [H, W, C] to [C, H, W]

    labels = np.array(labels)[:bboxes.shape[0]]

    hmap = make_heatmap(bboxes, labels, max_objs, num_classes=4)
    sequence = np.arange(max_objs)
    sequence[bboxes.shape[0]:] = -1

    # Collect syllables
    syllables = []
    for f in file_list:
        syllable = f.stem
        if syllable[0] in 'HR':
            syllable = syllable[1:] + syllable[0]
        elif syllable[0].isascii() and syllable[0].isalpha():
            syllable += 'L'
        syllables.append(syllable)
    syllables = syllables[:bboxes.shape[0]]

    bboxes = np.concatenate([bboxes, np.zeros((max_objs - bboxes.shape[0], 2, 2))], axis=0)

    return {
        'image': image,
        'bboxes': bboxes,
        'syllables': '.'.join(syllables),
        'sequence': sequence,
        **hmap
    }


def hstack_lines(lines):
    max_height = max(l[0].shape[0] for l in lines)
    offset = 0
    images = []
    all_coords = []
    for l, coords in lines[::-1]:
        images.append(np.pad(l, ((0, max_height - l.shape[0]), (0, 0), (0, 0)), constant_values=255))
        all_coords.append(coords + np.array([offset, 0]))
        offset += l.shape[1]
    return np.hstack(images), np.concatenate(all_coords[::-1])


def hstack_lines_resize(lines):
    max_height = max(l[0].shape[0] for l in lines)
    offset = 0
    images = []
    all_coords = []
    for l, coords in lines[::-1]:
        images.append(cv2.resize(l, dsize=(l.shape[1], max_height), interpolation=cv2.INTER_AREA))
        all_coords.append(coords * np.array([1, max_height / l.shape[0]]) + np.array([offset, 0]))
        offset += l.shape[1]
    return np.hstack(images), np.concatenate(all_coords[::-1])


def elastic_transform(image, bboxes, alpha, sigma, random_state=None):
    if random_state is None:
        random_state = np.random.RandomState(None)

    shape = image.shape
    N = 256
    mult = N / shape[0]
    noise_shape = [N, N * shape[1] // shape[0], 1]
    dx = alpha * gaussian_filter((random_state.rand(*noise_shape) * 2 - 1), sigma, mode="constant", cval=0)
    dy = alpha * gaussian_filter((random_state.rand(*noise_shape) * 2 - 1), sigma, mode="constant", cval=0)
    dz = np.zeros_like(dx)

    # find rects
    rects = np.zeros((noise_shape[0], noise_shape[1], bboxes.shape[0]))
    for i, (tl, br) in enumerate(bboxes * mult):
        rects[math.floor(tl[1]):math.floor(br[1]), math.ceil(tl[0]):math.ceil(br[0]), i] = 1

    x, y, z = np.meshgrid(np.arange(noise_shape[1]), np.arange(noise_shape[0]), np.arange(bboxes.shape[0]))
    indices = np.reshape(y + dy * mult, (-1, 1)), np.reshape(x + dx * mult, (-1, 1)), np.reshape(z, (-1, 1))

    distored_rects = map_coordinates(rects, indices, order=1, mode='constant', cval=0, prefilter=False)
    distored_rects = distored_rects.reshape(rects.shape)

    bounds = []
    for i in range(distored_rects.shape[-1]):
        x, y, w, h = cv2.boundingRect(np.uint8(distored_rects[..., i] > 0.5))
        bounds.append(np.array([[x, y], [x + w, y + h]]))
    bounds = np.stack(bounds) / mult

    # distort image
    x, y, z = np.meshgrid(np.arange(shape[1]), np.arange(shape[0]), np.arange(shape[2]))
    xx = x + cv2.resize(dx, dsize=[shape[1], shape[0]], interpolation=cv2.INTER_CUBIC)[..., None]
    yy = y + cv2.resize(dy, dsize=[shape[1], shape[0]], interpolation=cv2.INTER_CUBIC)[..., None]
    indices = np.reshape(yy, (-1, 1)), np.reshape(xx, (-1, 1)), np.reshape(z, (-1, 1))

    distored_image = map_coordinates(image, indices, order=1, mode='constant', cval=255, prefilter=False)
    distored_image = distored_image.reshape(image.shape)

    return distored_image, bounds


def make_heatmap(bboxes, labels, max_objs, num_classes,
                 fmap_size=None, img_size=None, gaussian=True, gaussian_iou=0.7):
    if fmap_size is None:
        fmap_size = {'h': 128, 'w': 128}
    if img_size is None:
        img_size = {'h': 512, 'w': 512}

    hmap_tl = np.zeros((num_classes, fmap_size['h'], fmap_size['w']), dtype=np.float32)
    hmap_br = np.zeros((num_classes, fmap_size['h'], fmap_size['w']), dtype=np.float32)
    hmap_ct = np.zeros((num_classes, fmap_size['h'], fmap_size['w']), dtype=np.float32)

    regs_tl = np.zeros((max_objs, 2), dtype=np.float32)
    regs_br = np.zeros((max_objs, 2), dtype=np.float32)
    regs_ct = np.zeros((max_objs, 2), dtype=np.float32)

    inds_tl = np.zeros((max_objs,), dtype=np.int64)
    inds_br = np.zeros((max_objs,), dtype=np.int64)
    inds_ct = np.zeros((max_objs,), dtype=np.int64)

    num_objs = np.array(min(bboxes.shape[0], max_objs))
    ind_masks = np.zeros((max_objs,), dtype=np.uint8)
    ind_masks[:num_objs] = 1

    for i, (((xtl, ytl), (xbr, ybr)), label) in enumerate(zip(bboxes, labels)):
        xct, yct = (xbr + xtl) / 2., (ybr + ytl) / 2.

        fxtl = (xtl * fmap_size['w'] / img_size['w'])
        fytl = (ytl * fmap_size['h'] / img_size['h'])
        fxbr = (xbr * fmap_size['w'] / img_size['w'])
        fybr = (ybr * fmap_size['h'] / img_size['h'])
        fxct = (xct * fmap_size['w'] / img_size['w'])
        fyct = (yct * fmap_size['h'] / img_size['h'])

        ixtl = min(max(int(fxtl), 0), fmap_size['w'] - 1)
        iytl = min(max(int(fytl), 0), fmap_size['h'] - 1)
        ixbr = min(max(int(fxbr), 0), fmap_size['w'] - 1)
        iybr = min(max(int(fybr), 0), fmap_size['h'] - 1)
        ixct = min(max(int(fxct), 0), fmap_size['w'] - 1)
        iyct = min(max(int(fyct), 0), fmap_size['h'] - 1)

        if gaussian:
            width = xbr - xtl
            height = ybr - ytl

            width = math.ceil(width * fmap_size['w'] / img_size['w'])
            height = math.ceil(height * fmap_size['h'] / img_size['h'])

            radius = max(0, int(gaussian_radius((height, width), gaussian_iou)))

            draw_gaussian(hmap_tl[label], [ixtl, iytl], radius)
            draw_gaussian(hmap_br[label], [ixbr, iybr], radius)
            draw_gaussian(hmap_ct[label], [ixct, iyct], radius, delta=5)
        else:
            hmap_tl[label, iytl, ixtl] = 1
            hmap_br[label, iybr, ixbr] = 1
            hmap_ct[label, iyct, ixct] = 1

        regs_tl[i, :] = [fxtl - ixtl, fytl - iytl]
        regs_br[i, :] = [fxbr - ixbr, fybr - iybr]
        regs_ct[i, :] = [fxct - ixct, fyct - iyct]
        inds_tl[i] = iytl * fmap_size['w'] + ixtl
        inds_br[i] = iybr * fmap_size['w'] + ixbr
        inds_ct[i] = iyct * fmap_size['w'] + ixct

    return {'hmap_tl': hmap_tl, 'hmap_br': hmap_br, 'hmap_ct': hmap_ct,
            'regs_tl': regs_tl, 'regs_br': regs_br, 'regs_ct': regs_ct,
            'inds_tl': inds_tl, 'inds_br': inds_br, 'inds_ct': inds_ct,
            'ind_masks': ind_masks}


def load_dataset(train_size=100000):
    df1 = list(pathlib.Path("img/img1").iterdir())
    df2 = list(pathlib.Path("img/img2").iterdir())
    df3 = list(pathlib.Path("img/img3").iterdir())
    df_hnaja1 = list(pathlib.Path("img/img-hanja1").iterdir())
    df_hanja2 = list(pathlib.Path("img/img-hanja2").iterdir())

    with open("annotations.json", "r") as fp:
        real_annotations = json.load(fp)

    train_df1, test_df1 = train_test_split(df1, test_size=128, random_state=0)
    train_df2, test_df2 = train_test_split(df2, test_size=128, random_state=0)
    train_df3, test_df3 = train_test_split(df3, test_size=128, random_state=0)
    train_dfh1, test_dfh1 = train_test_split(df_hnaja1, test_size=128, random_state=0)
    train_dfh2, test_dfh2 = train_test_split(df_hanja2, test_size=128, random_state=0)
    train_real_annotations, test_real_annotations = train_test_split(real_annotations, test_size=2, random_state=0)

    # Augmentation
    colorize = albumentations.RGBShift(r_shift_limit=0, g_shift_limit=0, b_shift_limit=[-80, 0])

    def color_get_params():
        a = random.uniform(-40, 0)
        b = random.uniform(-80, -30)
        return {"r_shift": a,
                "g_shift": a,
                "b_shift": b}

    colorize.get_params = color_get_params

    transform = albumentations.Compose([
        albumentations.CropAndPad(
            percent=0.05,
            pad_mode=cv2.BORDER_REPLICATE,
        ),
        albumentations.Perspective(
            always_apply=True,
            fit_output=True,
            pad_val=[255, 255, 255],
            scale=(0.0, 0.05),
        ),
        albumentations.Rotate(
            always_apply=True,
            limit=5,
            border_mode=cv2.BORDER_CONSTANT,
            value=[255, 255, 255]
        ),
        albumentations.Blur(blur_limit=(2, 5)),
        colorize,
        albumentations.GaussNoise(var_limit=(200.0, 200.0)),
    ],
        bbox_params=albumentations.BboxParams(
            format='coco', label_fields=['class_labels']
        ),
    )

    def augment(img, coords):
        # Apply Augmentations
        aug = transform(
            image=img,
            bboxes=np.concatenate([coords[:, 0], coords[:, 1] - coords[:, 0]], axis=1),
            class_labels=['hangul'] * len(coords),
        )

        bboxes = np.array(aug['bboxes'])
        aug_coords = np.stack([bboxes[:, :2], bboxes[:, :2] + bboxes[:, 2:]], axis=1)
        dimage, bounds = elastic_transform(aug['image'], aug_coords, 4000, 10)

        return dimage, bounds

    train_dataset = SyllableDataset(
        [train_df1, train_df2, train_df3],
        [train_dfh1, train_dfh2],
        train_real_annotations,
        augment, train_size
    )
    test_dataset = SyllableDataset(
        [test_df1, test_df2, test_df3],
        [test_dfh1, test_dfh2],
        test_real_annotations,
        None, 256
    )

    return train_dataset, test_dataset
