import torch
from torch.nn import CrossEntropyLoss

from transformers import VisionEncoderDecoderModel
from transformers import TrOCRProcessor, RobertaTokenizerFast


class SyllableRecognizer:
    def __init__(self, model=None):
        if model is None:
            self.model: VisionEncoderDecoderModel = VisionEncoderDecoderModel.from_pretrained(
                "ckpt-syllable-3fonts-surrounded-real"
            )
        else:
            self.model: VisionEncoderDecoderModel = model

        self.processor = TrOCRProcessor.from_pretrained("Chomkwoy/nilkessye_tokenizer")

    def _preprocess_images(self, images):
        pixel_values = []
        for image in images:
            pixel_values.append(self.processor(image, return_tensors="pt").pixel_values)
        pixel_values = torch.cat(pixel_values, dim=0)
        return pixel_values

    def recognize(self, images):
        pixel_values = self._preprocess_images(images)

        generated_ids = self.model.generate(
            pixel_values.to(self.model.device),
            max_new_tokens=13,
            early_stopping=True,
            eos_token_id=self.processor.tokenizer.eos_token_id
        )
        generated_text = self.processor.batch_decode(generated_ids, skip_special_tokens=True)
        return generated_text

    def loss(self, images, text):
        pixel_values = self._preprocess_images(images)
        tokens = self.processor.tokenizer(text, padding=True, return_tensors='pt')
        labels = tokens['input_ids']
        labels[labels == self.processor.tokenizer.pad_token_id] = -100

        with torch.no_grad():
            outputs = self.model(
                pixel_values=pixel_values.to(self.model.device),
                labels=labels.to(self.model.device),
                return_dict=True,
            )

        logits = outputs.logits.cpu()
        loss_fct = CrossEntropyLoss(reduction='none')
        loss = loss_fct(logits.permute(0, 2, 1), labels)

        return loss.sum(-1)
