import copy
import itertools

import cv2
import matplotlib.pyplot as plt
import numpy as np
import torch
from scipy.signal import find_peaks
from scipy.sparse.csgraph import floyd_warshall
from scipy.spatial import distance
from tqdm.auto import tqdm

from utils.keypoint import _decode


def get_pred_detections(output, sw, sh, threshold=0.4, ae_threshold=1.0, max_objs=9 * 16 * 4 * 2):
    detections, centers, seq_pred = _decode(
        *output[-1], ae_threshold=ae_threshold, K=max_objs, kernel=3, num_dets=100000)

    detections = detections.reshape(detections.shape[0], -1, 8).detach().cpu().numpy()
    detections = detections.reshape(-1, 8)
    detections = detections[detections[:, 4] > 0]

    centers = centers.reshape(centers.shape[0], -1, 4).detach().cpu().numpy()
    centers = centers.reshape(-1, 4)

    seq_pred = seq_pred[0].detach().cpu().numpy()

    # find matching rect for each center point
    # detections: [num_rects, 8 (tlx, tly, brx, bry, score, tlscore, brscore, cls)]
    # centers: [num_centers, 4 (x, y, cls, score)]
    detection_centers = np.stack([
        (detections[:, 0] + detections[:, 2]) / 2,
        (detections[:, 1] + detections[:, 3]) / 2
    ], axis=1)
    ratios = (detections[:, 3] - detections[:, 1]) / (detections[:, 2] - detections[:, 0])

    dist = distance.cdist(centers[:, :2], detection_centers)  # [num_centers, num_rects]
    tlx, brx = detections[:, 0][None, :], detections[:, 2][None, :]
    tly, bry = detections[:, 1][None, :], detections[:, 3][None, :]
    inside = (
            ((tlx * 0.7 + brx * 0.3) < centers[:, 0][:, None]) & (centers[:, 0][:, None] < (tlx * 0.3 + brx * 0.7)) &
            ((tly * 0.7 + bry * 0.3) < centers[:, 1][:, None]) & (centers[:, 1][:, None] < (tly * 0.3 + bry * 0.7))
    )

    scores = (
            -dist * .5  # penalize far center point
            + detections[None, :, 4] * 10  # original detection score
            + inside * 100  # enforce center point inside the bounding box
            + (1 - (ratios > 2.0)) * 100  # dont select too tall boxes
            + (1 - (ratios < 0.2)) * 100  # dont select too wide boxes
            - (brx - tlx) * (bry - tly) * 0.02  # prefer smaller boxes
    )
    rect_idxs = np.argsort(scores, axis=1)[:, ::-1]

    tiles = []
    for (x, y, cs, score), idxs, seq in zip(centers, rect_idxs, seq_pred):
        for i in idxs[0:1]:
            tlx, tly, brx, bry = detections[i, :4]
            rx, ry = (x - tlx) / (brx - tlx), (y - tly) / (bry - tly)
            if score > threshold and 0.3 < rx < 0.7 and 0.3 < ry < 0.7:
                bbox = (
                    (int(tlx * sw), int(tly * sh)),
                    (int(brx * sw), int(bry * sh))
                )
                cx, cy = int(x * sw), int(y * sh)
                tiles.append((bbox, (cx, cy), seq, cs, score))

    tiles = sorted(tiles, key=lambda tile: tile[4], reverse=True)

    filtered_tiles = []
    for bbox, (cx, cy), seq, cs, score in tiles:
        max_iou = max((bb_intersection_over_union(bbox, bbox2) for bbox2, _, _, _ in filtered_tiles), default=0)
        if max_iou < 0.90:
            filtered_tiles.append((bbox, (cx, cy), seq, cs))

    tiles = filtered_tiles

    tiles = sorted(tiles, key=lambda tile: tile[2])

    return tiles


def sigmoid(z):
    return 1.0 / (1.0 + np.exp(-z))


def get_center(bbox):
    (tlx, tly), (brx, bry) = bbox
    return (tlx + brx) / 2, (tly + bry) / 2


def bb_intersection_over_union(boxA, boxB):
    # determine the (x, y)-coordinates of the intersection rectangle
    xA = max(boxA[0][0], boxB[0][0])
    yA = max(boxA[0][1], boxB[0][1])
    xB = min(boxA[1][0], boxB[1][0])
    yB = min(boxA[1][1], boxB[1][1])
    # compute the area of intersection rectangle
    interArea = max(0, xB - xA + 1) * max(0, yB - yA + 1)
    # compute the area of both the prediction and ground-truth
    # rectangles
    boxAArea = (boxA[1][0] - boxA[0][0] + 1) * (boxA[1][1] - boxA[0][1] + 1)
    boxBArea = (boxB[1][0] - boxB[0][0] + 1) * (boxB[1][1] - boxB[0][1] + 1)
    # compute the intersection over union by taking the intersection
    # area and dividing it by the sum of prediction + ground-truth
    # areas - the interesection area
    iou = interArea / float(boxAArea + boxBArea - interArea)
    # return the intersection over union value
    return iou


def batched(iterable, n):
    """Batch data into lists of length n. The last batch may be shorter."""
    # batched('ABCDEFG', 3) --> ABC DEF G
    it = iter(iterable)
    while True:
        batch = list(itertools.islice(it, n))
        if not batch:
            return
        yield batch


def find_line_angle(
        cur_centers,
        cur_bboxes,
        k=5,
        n_bins=365,  # per 180 degrees
        verbose=False
):
    N = len(cur_centers)

    if N == 0:
        return None

    bbox_heights = np.array([bry - tly for (tlx, tly), (brx, bry) in cur_bboxes])

    corners = np.stack([
        cur_bboxes[:, 0, :],  # tl
        np.stack([cur_bboxes[:, 0, 0], cur_bboxes[:, 1, 1]], axis=-1),  # bl
        np.stack([cur_bboxes[:, 1, 0], cur_bboxes[:, 0, 1]], axis=-1),  # tr
        cur_bboxes[:, 1, :],  # br
    ], axis=1)

    dist_matrix = distance.cdist(corners.reshape(-1, 2), corners.reshape(-1, 2))
    dist_matrix = dist_matrix.reshape((N, 4, N, 4)).transpose(0, 2, 1, 3)  # [N, N, 4, 4]
    dist_matrix = dist_matrix.min(axis=(2, 3))
    np.fill_diagonal(dist_matrix, 1e9)
    k_nearest_neighbors_indices = np.argsort(dist_matrix, axis=1)[:, :k]

    # Find line angle
    k_nearest_neighbors = cur_centers[k_nearest_neighbors_indices]

    diff = (k_nearest_neighbors - cur_centers[:, None, :])
    angles = np.fmod(np.arctan2(diff[..., 1], diff[..., 0]) + np.pi * 2, np.pi)

    angle_histogram, bin_edges = np.histogram(angles.flatten(), bins=n_bins)
    angle_histogram = angle_histogram.astype(float)

    # Avoid finding horizontal lines
    angle_histogram[0:n_bins // 4] *= 0.5
    angle_histogram[-n_bins // 4:] *= 0.5

    # Wrap angle
    angle_histogram = np.concatenate([angle_histogram, angle_histogram])

    # smoothing filter
    window_size = n_bins // 16
    box = np.ones(window_size) / window_size
    angle_histogram = np.convolve(angle_histogram, box, mode='same')

    # find biggest peak
    peaks, properties = find_peaks(angle_histogram, prominence=0.5, width=4)

    if verbose:
        plt.plot(angle_histogram)
        plt.plot(peaks, angle_histogram[peaks], "x")
        plt.vlines(x=peaks, ymin=angle_histogram[peaks] - properties["prominences"],
                   ymax=angle_histogram[peaks], color="C1")
        plt.hlines(y=properties["width_heights"], xmin=properties["left_ips"],
                   xmax=properties["right_ips"], color="C1")
        plt.show()

    if len(peaks) == 0:
        return None

    peak_bin = [peak_pos for _, peak_pos in sorted(zip(properties["prominences"], peaks))][-1]
    line_angle = np.fmod(peak_bin * np.pi / n_bins, np.pi)

    return line_angle


def find_lines(
        cur_centers,
        cur_bboxes,
        line_angle,
        center_dist_threshold=2.,
        corner_dist_threshold=0.5,
        k=7,
        angle_delta=30 * (np.pi / 180),
):
    N = len(cur_centers)

    if N == 0:
        return [], np.zeros((0, k))

    bbox_heights = np.array([bry - tly for (tlx, tly), (brx, bry) in cur_bboxes])
    mean_bbox_height = bbox_heights.mean()

    corners = np.stack([
        cur_bboxes[:, 0, :],  # tl
        np.stack([cur_bboxes[:, 0, 0], cur_bboxes[:, 1, 1]], axis=-1),  # bl
        np.stack([cur_bboxes[:, 1, 0], cur_bboxes[:, 0, 1]], axis=-1),  # tr
        cur_bboxes[:, 1, :],  # br
    ], axis=1)

    corner_dist_matrix = distance.cdist(corners.reshape((-1, 2)), corners.reshape((-1, 2)))
    corner_dist_matrix = corner_dist_matrix.reshape((N, 4, N, 4)).transpose(0, 2, 1, 3)
    corner_dist_matrix = corner_dist_matrix.min(axis=(2, 3))
    np.fill_diagonal(corner_dist_matrix, 1e9)

    dist_matrix = distance.cdist(cur_centers, cur_centers)
    np.fill_diagonal(dist_matrix, 1e9)
    k_nearest_neighbors_indices = np.argsort(dist_matrix, axis=1)[:, :k]
    k_nearest_neighbors = cur_centers[k_nearest_neighbors_indices]

    k_nearest_neighbors_dists = dist_matrix[np.arange(N)[:, None], k_nearest_neighbors_indices]
    k_nearest_neighbors_corner_dists = corner_dist_matrix[np.arange(N)[:, None], k_nearest_neighbors_indices]

    diff = (k_nearest_neighbors - cur_centers[:, None, :])
    angles = np.fmod(np.arctan2(diff[..., 1], diff[..., 0]) + np.pi * 2, np.pi)

    # Make inline & between-line neighbor graphs
    line_range = (line_angle - angle_delta, line_angle + angle_delta)
    is_inline = (
            ((line_range[0] < angles) & (angles < line_range[1])) |
            ((line_range[0] - np.pi < angles) & (angles < line_range[1] - np.pi)) |
            ((line_range[0] + np.pi < angles) & (angles < line_range[1] + np.pi))
    )

    inline_neighbors_indices = k_nearest_neighbors_indices.copy()
    inline_neighbors_indices[~is_inline] = -1
    inline_neighbors_indices[k_nearest_neighbors_dists > mean_bbox_height * center_dist_threshold] = -1
    inline_neighbors_indices[k_nearest_neighbors_corner_dists > mean_bbox_height * corner_dist_threshold] = -1

    def transitive_closure(neighbor_indices):
        reachable = np.zeros((N, N))
        reachable[:, :] = 1e9
        for i in range(N):
            for j in neighbor_indices[i]:
                if j != -1:
                    reachable[i, j] = reachable[j, i] = 1
        reachable = floyd_warshall(reachable, directed=False)
        reachable = reachable < 1e9

        groups = []

        visited = np.zeros((N,))
        for i in range(N):
            if visited[i]:
                continue
            group = np.nonzero(reachable[i])[0]
            visited[group] = 1
            groups.append(group)

        return groups

    lines = transitive_closure(inline_neighbors_indices)

    return lines, inline_neighbors_indices


def detect_lines(tiles):
    main_tiles = [(bbox, center, seq, cls) for bbox, center, seq, cls in tiles if cls in [0, 1]]
    anno_tiles = [(bbox, center, seq, cls) for bbox, center, seq, cls in tiles if cls in [2, 3]]

    main_centers = np.array([center for bbox, center, seq, cls in tiles if cls in [0, 1]]).reshape(-1, 2)
    anno_centers = np.array([center for bbox, center, seq, cls in tiles if cls in [2, 3]]).reshape(-1, 2)

    main_bboxes = np.array([bbox for bbox, center, seq, cls in tiles if cls in [0, 1]]).reshape(-1, 2, 2)
    anno_bboxes = np.array([bbox for bbox, center, seq, cls in tiles if cls in [2, 3]]).reshape(-1, 2, 2)

    # Find line angle
    main_line_angle = find_line_angle(main_centers, main_bboxes)
    anno_line_angle = find_line_angle(anno_centers, anno_bboxes)

    line_angles = []
    if main_line_angle is not None:
        line_angles.append((main_line_angle, len(main_centers)))
    if anno_line_angle is not None:
        # wrap angle
        if main_line_angle is not None:
            anno_line_angles = np.array([anno_line_angle, anno_line_angle - np.pi, anno_line_angle + np.pi])
            anno_line_angle = anno_line_angles[np.abs(anno_line_angles - main_line_angle).argmin()]
        line_angles.append((anno_line_angle, len(anno_centers)))

    denominator = sum(n for _, n in line_angles)
    line_angle = sum(angle * (n / denominator) for angle, n in line_angles)
    line_angle = np.fmod(line_angle + np.pi * 2, np.pi)

    main_lines, main_inline_neighbors_indices = find_lines(
        main_centers, main_bboxes, line_angle,
        center_dist_threshold=2,
        corner_dist_threshold=0.2,
    )
    anno_lines, anno_inline_neighbors_indices = find_lines(
        anno_centers, anno_bboxes, line_angle,
        center_dist_threshold=1.4,
        corner_dist_threshold=0.7,
    )

    main_lines = [[main_tiles[i] for i in line] for line in main_lines]
    anno_lines = [[anno_tiles[i] for i in line] for line in anno_lines]

    all_lines = main_lines + anno_lines

    # Sort syllable in each line by increasing center y coord
    all_lines = [
        sorted(line, key=lambda tile: tile[1][1])
        for line in all_lines
    ]

    # Sort lines
    def seq_score(line):
        start_x = np.array([bbox[1][0] for bbox, center, seq, cls in line]).min()
        start_y = np.array([bbox[0][1] for bbox, center, seq, cls in line]).min()
        return start_y * 0.1 - start_x

    all_lines = sorted(all_lines, key=seq_score)

    line_infos = []
    for line in all_lines:
        tlx = np.array([bbox[0][0] for bbox, center, seq, cls in line]).mean()
        tly = np.array([bbox[0][1] for bbox, center, seq, cls in line]).min()
        brx = np.array([bbox[1][0] for bbox, center, seq, cls in line]).mean()
        bry = np.array([bbox[1][1] for bbox, center, seq, cls in line]).max()
        line_bbox = ((tlx, tly), (brx, bry))
        is_anno = line[0][3] in [2, 3]
        line_infos.append({
            'line': line,
            'bbox': line_bbox,
            'is_anno': is_anno,
        })

    # Sort lines by actual reading order
    line_infos = sort_lines(line_infos)

    return line_infos


def sort_lines(line_infos):
    lines_left = copy.copy(line_infos)
    ordered_lines = [lines_left[0]]
    del lines_left[0]
    anno_line_num = 0

    def dist(a, b):
        return np.sqrt((a[0] - b[0]) ** 2 + (a[1] - b[1]) ** 2)

    while len(lines_left) > 0:
        cur_line = ordered_lines[-1]
        (tlx, tly), (brx, bry) = cur_line['bbox']
        line_width = (brx - tlx)

        if cur_line['is_anno']:

            if anno_line_num == 0:
                # check if there's a second anno line
                distances = [
                    (dist((tlx, tly), (cand['bbox'][1][0], cand['bbox'][0][1])), i)
                    for i, cand in enumerate(lines_left)
                    if cand['is_anno']
                ]
                min_dist, min_idx = min(distances, default=(1e9, None))

                if min_dist < line_width / 2:
                    ordered_lines.append(lines_left[min_idx])
                    del lines_left[min_idx]
                    # print('anno->anno')
                    anno_line_num += 1
                    continue

                next_expected_tr = (brx, bry)

            else:  # anno_line_num == 1
                next_expected_tr = (brx + line_width, bry)

            # check for next main line
            distances = [
                (dist(next_expected_tr, (cand['bbox'][1][0], cand['bbox'][0][1])), i)
                for i, cand in enumerate(lines_left)
                if not cand['is_anno']
            ]

            min_dist, min_idx = min(distances, default=(1e9, None))

            if min_dist < line_width:
                ordered_lines.append(lines_left[min_idx])
                del lines_left[min_idx]
                # print('anno->main')
                anno_line_num = 0
                continue

            # select next line
            ordered_lines.append(lines_left[0])
            del lines_left[0]

        else:  # not cur_line['is_anno']

            # check for next anno line
            distances = [
                (dist((brx, bry), (cand['bbox'][1][0], cand['bbox'][0][1])), i)
                for i, cand in enumerate(lines_left)
                if cand['is_anno']
            ]

            min_dist, min_idx = min(distances, default=(1e9, None))

            if min_dist < line_width / 2:
                ordered_lines.append(lines_left[min_idx])
                del lines_left[min_idx]
                # print('main->anno', min_idx)
                anno_line_num = 0
                continue

            # select next line
            # print('main->main')
            ordered_lines.append(lines_left[0])
            del lines_left[0]

    return ordered_lines


def recognize_lines(line_infos, orig_image, syllable_recognizer, batch_size=32):
    tiles = []
    for line_idx, line_info in enumerate(line_infos):
        for bbox, center, seq, cls in line_info['line']:
            (tlx, tly), (brx, bry) = bbox
            w, h = brx - tlx, bry - tly
            pw, ph = w / 5, h / 5
            tile = orig_image[
                max(0, int(tly - ph)):min(orig_image.shape[0], int(bry + ph)),
                max(0, int(tlx - pw)):min(orig_image.shape[1], int(brx + pw)),
            ]
            tiles.append((tile, bbox, center, seq, cls))

    hangul_tiles = [(i, tile) for i, (tile, _, _, _, cls) in enumerate(tiles) if cls in [0, 2]]

    pred_syllables = ["〓"] * len(tiles)
    batches = list(batched(hangul_tiles, batch_size))
    for batch in tqdm(batches):
        indices, images = zip(*batch)
        batch_pred_syllables = syllable_recognizer.recognize(images)
        for i, pred_syllable in zip(indices, batch_pred_syllables):
            pred_syllables[i] = pred_syllable

    return pred_syllables


def recognize_page(orig_image, centernet, syllable_recognizer, return_line_infos=False, batch_size=32):
    orig_size = (orig_image.shape[1], orig_image.shape[0])
    image = cv2.resize(orig_image, dsize=(512, 512), interpolation=cv2.INTER_AREA)

    image = image.astype(np.float32) / 255. - .5  # to [-.5, +.5] range
    image = image.transpose((2, 0, 1))  # [H, W, C] to [C, H, W]
    image = torch.as_tensor(image)

    # Run object detection
    centernet.eval()
    with torch.no_grad():
        output = centernet(torch.as_tensor(image)[None].to(centernet.device))

    sw, sh = orig_size[0] * 4 / 512, orig_size[1] * 4 / 512

    tiles = get_pred_detections(
        output, sw=sw, sh=sh,
        threshold=0.3,
        ae_threshold=20.0
    )

    line_infos = detect_lines(tiles)

    pred_syllables = recognize_lines(line_infos, orig_image, syllable_recognizer, batch_size=batch_size)

    if return_line_infos:
        return pred_syllables, line_infos

    return pred_syllables
