import torch
import torch.nn as nn
import torch.nn.functional as F
from cpool_new import TopPool, BottomPool, LeftPool, RightPool


class pool(nn.Module):
    def __init__(self, dim, pool1, pool2):
        super(pool, self).__init__()
        self.p1_conv1 = convolution(3, dim, 128)
        self.p2_conv1 = convolution(3, dim, 128)

        self.p_conv1 = nn.Conv2d(128, dim, 3, padding=1, bias=False)
        self.p_bn1 = nn.BatchNorm2d(dim)

        self.conv1 = nn.Conv2d(dim, dim, 1, bias=False)
        self.bn1 = nn.BatchNorm2d(dim)

        self.conv2 = convolution(3, dim, dim)

        self.pool1 = pool1()
        self.pool2 = pool2()

        self.look_conv1 = convolution(3, dim, 128)
        self.look_conv2 = convolution(3, dim, 128)
        self.p1_look_conv = nn.Conv2d(128, 128, 3, padding=1, bias=False)
        self.p2_look_conv = nn.Conv2d(128, 128, 3, padding=1, bias=False)

    def forward(self, x):
        pool1 = self.pool1(self.p1_look_conv(self.p1_conv1(x) +
                                             self.pool2(self.look_conv1(x))))
        pool2 = self.pool2(self.p2_look_conv(self.p2_conv1(x) +
                                             self.pool1(self.look_conv2(x))))

        p_bn1 = self.p_bn1(self.p_conv1(pool1 + pool2))
        bn1 = self.bn1(self.conv1(x))

        out = self.conv2(F.relu(p_bn1 + bn1, inplace=True))
        return out


class pool_cross(nn.Module):
    def __init__(self, dim):
        super(pool_cross, self).__init__()
        self.p1_conv1 = convolution(3, dim, 128)
        self.p2_conv1 = convolution(3, dim, 128)

        self.p_conv1 = nn.Conv2d(128, dim, 3, padding=1, bias=False)
        self.p_bn1 = nn.BatchNorm2d(dim)

        self.conv1 = nn.Conv2d(dim, dim, 1, bias=False)
        self.bn1 = nn.BatchNorm2d(dim)

        self.conv2 = convolution(3, dim, dim)

        self.pool_top = TopPool()
        self.pool_left = LeftPool()
        self.pool_bottom = BottomPool()
        self.pool_right = RightPool()

    def forward(self, x):
        # pool 1
        pool1 = self.pool_bottom(self.pool_top(self.p1_conv1(x)))

        # pool 2
        pool2 = self.pool_right(self.pool_left(self.p2_conv1(x)))

        # pool 1 + pool 2
        p_bn1 = self.p_bn1(self.p_conv1(pool1 + pool2))
        bn1 = self.bn1(self.conv1(x))

        out = self.conv2(F.relu(p_bn1 + bn1, inplace=True))
        return out


class convolution(nn.Module):
    def __init__(self, k, inp_dim, out_dim, stride=1, with_bn=True):
        super(convolution, self).__init__()

        pad = (k - 1) // 2
        self.conv = nn.Conv2d(inp_dim, out_dim, (k, k), padding=(pad, pad), stride=(stride, stride), bias=not with_bn)
        self.bn = nn.BatchNorm2d(out_dim) if with_bn else nn.Sequential()
        self.relu = nn.ReLU(inplace=True)

    def forward(self, x):
        conv = self.conv(x)
        bn = self.bn(conv)
        relu = self.relu(bn)
        return relu


class residual(nn.Module):
    def __init__(self, k, inp_dim, out_dim, stride=1, with_bn=True):
        super(residual, self).__init__()

        self.conv1 = nn.Conv2d(inp_dim, out_dim, (3, 3), padding=(1, 1), stride=(stride, stride), bias=False)
        self.bn1 = nn.BatchNorm2d(out_dim)
        self.relu1 = nn.ReLU(inplace=True)

        self.conv2 = nn.Conv2d(out_dim, out_dim, (3, 3), padding=(1, 1), bias=False)
        self.bn2 = nn.BatchNorm2d(out_dim)

        self.skip = nn.Sequential(nn.Conv2d(inp_dim, out_dim, (1, 1), stride=(stride, stride), bias=False),
                                  nn.BatchNorm2d(out_dim)) \
            if stride != 1 or inp_dim != out_dim else nn.Sequential()
        self.relu = nn.ReLU(inplace=True)

    def forward(self, x):
        conv1 = self.conv1(x)
        bn1 = self.bn1(conv1)
        relu1 = self.relu1(bn1)

        conv2 = self.conv2(relu1)
        bn2 = self.bn2(conv2)

        skip = self.skip(x)
        return self.relu(bn2 + skip)


# inp_dim -> out_dim -> ... -> out_dim
def make_layer(kernel_size, inp_dim, out_dim, modules, layer, stride=1):
    layers = [layer(kernel_size, inp_dim, out_dim, stride=stride)]
    layers += [layer(kernel_size, out_dim, out_dim) for _ in range(modules - 1)]
    return nn.Sequential(*layers)


# inp_dim -> inp_dim -> ... -> inp_dim -> out_dim
def make_layer_revr(kernel_size, inp_dim, out_dim, modules, layer):
    layers = [layer(kernel_size, inp_dim, inp_dim) for _ in range(modules - 1)]
    layers.append(layer(kernel_size, inp_dim, out_dim))
    return nn.Sequential(*layers)


# key point layer
def make_kp_layer(cnv_dim, curr_dim, out_dim):
    return nn.Sequential(convolution(3, cnv_dim, curr_dim, with_bn=False),
                         nn.Conv2d(curr_dim, out_dim, (1, 1)))


class kp_module(nn.Module):
    def __init__(self, n, dims, modules):
        super(kp_module, self).__init__()

        self.n = n

        curr_modules = modules[0]
        next_modules = modules[1]

        curr_dim = dims[0]
        next_dim = dims[1]

        # Upper branch: Repeat curr_mod times residual, curr_dim -> curr_dim -> ... -> curr_dim
        self.top = make_layer(3, curr_dim, curr_dim, curr_modules, layer=residual)
        # The resolution should have been halved here...
        self.down = nn.Sequential()
        # Repeat curr_mod times residual, curr_dim -> next_dim -> ... -> next_dim
        # In fact, the resolution is reduced by the first convolutional layer here.
        self.low1 = make_layer(3, curr_dim, next_dim, curr_modules, layer=residual, stride=2)
        # There is still an hourglass in the middle of hourglass
        # Until the end of the recursion, repeat next_mod residual times, next_dim -> next_dim -> ... -> next_dim
        if self.n > 1:
            self.low2 = kp_module(n - 1, dims[1:], modules[1:])
        else:
            self.low2 = make_layer(3, next_dim, next_dim, next_modules, layer=residual)
        # Repeat curr_mod times residual, next_dim -> next_dim -> ... -> next_dim -> curr_dim
        self.low3 = make_layer_revr(3, next_dim, curr_dim, curr_modules, layer=residual)
        # Resolution here X 2
        self.up = nn.Upsample(scale_factor=2)

    def forward(self, x):
        up1 = self.top(x)  # upper branch residual
        down = self.down(x)  # Lower branch downsample (not merged)
        low1 = self.low1(down)  # Lower branch residual
        low2 = self.low2(low1)  # Lower branch hourglass
        low3 = self.low3(low2)  # Lower branch residual
        up2 = self.up(low3)  # Lower branch upsample
        return up1 + up2  # Merge upper and lower branches


class exkp(nn.Module):
    def __init__(self, n, nstack, dims, modules, num_classes=80, cnv_dim=256):
        super(exkp, self).__init__()

        self.nstack = nstack

        curr_dim = dims[0]

        self.pre = nn.Sequential(convolution(7, 3, 128, stride=2),
                                 residual(3, 128, curr_dim, stride=2))

        self.kps = nn.ModuleList([kp_module(n, dims, modules) for _ in range(nstack)])

        self.cnvs = nn.ModuleList([convolution(3, curr_dim, cnv_dim) for _ in range(nstack)])

        self.inters = nn.ModuleList([residual(3, curr_dim, curr_dim) for _ in range(nstack - 1)])

        self.inters_ = nn.ModuleList([nn.Sequential(nn.Conv2d(curr_dim, curr_dim, (1, 1), bias=False),
                                                    nn.BatchNorm2d(curr_dim))
                                      for _ in range(nstack - 1)])
        self.cnvs_ = nn.ModuleList([nn.Sequential(nn.Conv2d(cnv_dim, curr_dim, (1, 1), bias=False),
                                                  nn.BatchNorm2d(curr_dim))
                                    for _ in range(nstack - 1)])

        self.cnvs_tl = nn.ModuleList([pool(cnv_dim, TopPool, LeftPool) for _ in range(nstack)])
        self.cnvs_br = nn.ModuleList([pool(cnv_dim, BottomPool, RightPool) for _ in range(nstack)])
        self.cnvs_ct = nn.ModuleList([pool_cross(cnv_dim) for _ in range(nstack)])

        # heatmap layers
        self.hmap_tl = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, num_classes) for _ in range(nstack)])
        self.hmap_br = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, num_classes) for _ in range(nstack)])
        self.hmap_ct = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, num_classes) for _ in range(nstack)])

        # embedding layers
        self.embd_tl = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, 16) for _ in range(nstack)])
        self.embd_br = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, 16) for _ in range(nstack)])

        for hmap_tl, hmap_br, hmap_ct in zip(self.hmap_tl, self.hmap_br, self.hmap_ct):
            hmap_tl[-1].bias.data.fill_(-2.19)
            hmap_br[-1].bias.data.fill_(-2.19)
            hmap_ct[-1].bias.data.fill_(-2.19)

        # regression layers
        self.regs_tl = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, 2) for _ in range(nstack)])
        self.regs_br = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, 2) for _ in range(nstack)])

        # [offset x, offset y, sequence]
        self.regs_ct = nn.ModuleList([make_kp_layer(cnv_dim, curr_dim, 3) for _ in range(nstack)])

        self.relu = nn.ReLU(inplace=True)

    def forward(self, inputs):
        inter = self.pre(inputs)

        outs = []
        for ind in range(self.nstack):
            kp = self.kps[ind](inter)
            cnv = self.cnvs[ind](kp)

            if self.training or ind == self.nstack - 1:
                cnv_tl = self.cnvs_tl[ind](cnv)
                cnv_br = self.cnvs_br[ind](cnv)
                cnv_ct = self.cnvs_ct[ind](cnv)

                hmap_tl, hmap_br = self.hmap_tl[ind](cnv_tl), self.hmap_br[ind](cnv_br)
                embd_tl, embd_br = self.embd_tl[ind](cnv_tl), self.embd_br[ind](cnv_br)
                regs_tl, regs_br = self.regs_tl[ind](cnv_tl), self.regs_br[ind](cnv_br)

                hmap_ct = self.hmap_ct[ind](cnv_ct)
                regs_ct = self.regs_ct[ind](cnv_ct)

                outs.append([hmap_tl, hmap_br, hmap_ct, embd_tl, embd_br, regs_tl, regs_br, regs_ct])

            if ind < self.nstack - 1:
                inter = self.inters_[ind](inter) + self.cnvs_[ind](cnv)
                inter = self.relu(inter)
                inter = self.inters[ind](inter)
        return outs


def _tranpose_and_gather_feature(feature, ind):
    feature = feature.permute(0, 2, 3, 1).contiguous()  # [B, C, H, W] => [B, H, W, C]
    feature = feature.view(feature.size(0), -1, feature.size(3))  # [B, H, W, C] => [B, H x W, C]
    ind = ind[:, :, None].expand(ind.shape[0], ind.shape[1], feature.shape[-1])  # [B, num_obj] => [B, num_obj, C]
    feature = feature.gather(1, ind)  # [B, H x W, C] => [B, num_obj, C]
    return feature


def _neg_loss(preds, targets):
    pos_inds = targets == 1  # todo targets > 1-epsilon ?
    neg_inds = targets < 1  # todo targets < 1-epsilon ?

    neg_weights = torch.pow(1 - targets[neg_inds], 4)

    loss = 0
    for pred in preds:
        pred = torch.clamp(torch.sigmoid(pred), min=1e-4, max=1 - 1e-4)
        pos_pred = pred[pos_inds]
        neg_pred = pred[neg_inds]

        pos_loss = torch.log(pos_pred) * torch.pow(1 - pos_pred, 2)
        neg_loss = torch.log(1 - neg_pred) * torch.pow(neg_pred, 2) * neg_weights

        num_pos = pos_inds.float().sum()
        pos_loss = pos_loss.sum()
        neg_loss = neg_loss.sum()

        if pos_pred.nelement() == 0:
            loss = loss - neg_loss
        else:
            loss = loss - (pos_loss + neg_loss) / num_pos
    return loss / len(preds)


def _ae_loss(embd0s, embd1s, mask):
    num = mask.sum(dim=1, keepdim=True).unsqueeze(2).float()  # [B, 1]

    pull, push = 0, 0
    for embd0, embd1 in zip(embd0s, embd1s):  # [B, num_obj, vec_dim]
        embd_mean = (embd0 + embd1) / 2

        embd0 = (embd0 - embd_mean)**2 / (num + 1e-4)
        embd0 = embd0[mask].sum()
        embd1 = (embd1 - embd_mean)**2 / (num + 1e-4)
        embd1 = embd1[mask].sum()
        pull += embd0 + embd1

        push_mask = (mask[:, None, :] + mask[:, :, None]) == 2  # [B, num_obj, num_obj]
        dist = F.relu(1 - (embd_mean[:, None, :] - embd_mean[:, :, None]).abs(), inplace=True)
        dist = dist - 1 / (num[:, :, None] + 1e-4)  # substract diagonal elements
        dist = dist / ((num - 1) * num + 1e-4)[:, :, None]  # total num element is n*n-n
        push += dist[push_mask].sum()
    return pull / len(embd0s), push / len(embd0s)


def _reg_loss(regs, gt_regs, mask):
    num = mask.float().sum() + 1e-4
    mask = mask[:, :, None].expand_as(gt_regs)  # [B, num_obj, 2]
    loss = sum([F.smooth_l1_loss(r[mask], gt_regs[mask], reduction='sum') / num for r in regs])
    return loss / len(regs)


def spearman(pred, target, mask):
    import torchsort
    x = 1e-2
    pred = torchsort.soft_rank(pred, regularization_strength=x)     # [B, L]
    target = torchsort.soft_rank(target, regularization_strength=x)  # [B, L]
    pred = pred - (pred * mask).sum(1, keepdim=True) / mask.sum(1, keepdim=True)
    pred = pred / ((pred * mask)**2).sum(1, keepdim=True).sqrt()
    target = target - (target * mask).sum(1, keepdim=True) / mask.sum(1, keepdim=True)
    target = target / ((target * mask)**2).sum(1, keepdim=True).sqrt()

    return (pred * target * mask).sum(1)  # [B]


def compute_loss(preds, batch, return_dict=False):
    """
    batch: {
        'image' : [B, C, H, W]
        'hmap_tl', 'hmap_br', 'hmap_ct' : [n_cls, fH, fW].  Heatmap (gaussian).
        'embd_tl', 'embd_br'            : [fH, fW].         Corner embeddings
        'regs_tl', 'regs_br', 'regs_ct' : [max_objs, 2].    Offsets (compensating for discretization error). In [0, 1) range.
        'inds_tl', 'inds_br', 'inds_ct' : [max_objs] int64. Indices of heatmap pixel.
        'ind_masks': [max_objs] bool. How many objects are there.
    }
    """
    hmap_tl, hmap_br, hmap_ct, embd_tl, embd_br, regs_tl, regs_br, regs_ct_and_seq = zip(*preds)

    embd_tl = [_tranpose_and_gather_feature(e, batch['inds_tl']) for e in embd_tl]
    embd_br = [_tranpose_and_gather_feature(e, batch['inds_br']) for e in embd_br]
    regs_tl = [_tranpose_and_gather_feature(r, batch['inds_tl']) for r in regs_tl]
    regs_br = [_tranpose_and_gather_feature(r, batch['inds_br']) for r in regs_br]
    regs_ct = [_tranpose_and_gather_feature(r, batch['inds_ct'])[..., :2] for r in regs_ct_and_seq]
    seq_pred = [_tranpose_and_gather_feature(r, batch['inds_ct'])[..., 2] for r in regs_ct_and_seq]

    focal_loss = _neg_loss(hmap_tl, batch['hmap_tl']) + \
                 _neg_loss(hmap_br, batch['hmap_br']) + \
                 _neg_loss(hmap_ct, batch['hmap_ct'])
    reg_loss = _reg_loss(regs_tl, batch['regs_tl'], batch['ind_masks']) + \
               _reg_loss(regs_br, batch['regs_br'], batch['ind_masks']) + \
               _reg_loss(regs_ct, batch['regs_ct'], batch['ind_masks'])
    pull_loss, push_loss = _ae_loss(embd_tl, embd_br, batch['ind_masks'])

    loss = focal_loss + 0.1 * pull_loss + 0.1 * push_loss + reg_loss

    if 'sequence' in batch and 'sequence_valid' in batch:
        rank_loss = sum(
            torch.log(1 - spearman(
                s[batch['sequence_valid']],
                batch['sequence'][batch['sequence_valid']],
                batch['ind_masks'][batch['sequence_valid']]
            ) + 1e-6)
            for s in seq_pred
        ) / len(seq_pred)
        loss = loss + rank_loss * 0.1
    else:
        rank_loss = torch.tensor([0.0], device=loss.device)

    if return_dict:
        return loss, {
            'focal_loss': focal_loss,
            'pull_loss': pull_loss,
            'push_loss': push_loss,
            'reg_loss': reg_loss,
            'orig_loss': focal_loss + 0.1 * pull_loss + 0.1 * push_loss + reg_loss,
            'rank_loss': rank_loss,
        }

    return loss
