import glob
import json
import pathlib
import re
from collections import Counter

import Levenshtein
import cv2
import numpy as np
import pandas as pd
from matplotlib import pyplot as plt
from natsort import natsorted
from scipy.signal import find_peaks

from utils import hanja


def load_book(jsonfile, img_dir, imgstart=1):
    with open(jsonfile, 'r') as fp:
        texts = json.load(fp)

    print(f"Loading {jsonfile}...")

    page_numbers = []
    for s in texts:
        if 'page' not in s:
            continue
        if ('lang' in s and s['lang'] == 'chi' and
                'type' in s and s['type'] in ['main', 'anno', 'anno2', 'anno3']):
            continue
        pns = s['page'].split('-')
        page_numbers.extend(pns)

    occurred = set()
    unique_page_numbers = []
    for p in page_numbers:
        if p not in occurred:
            unique_page_numbers.append(p)
        occurred.add(p)
    page_numbers = unique_page_numbers

    print(f"Page numbers = {page_numbers}")

    pages = []
    page = 0

    img_files = glob.glob(f"{img_dir}/*.png")
    last_idx = int(pathlib.Path(natsorted(img_files)[-1]).stem)

    for i in range(imgstart, last_idx + 1):
        filename = f"{img_dir}/{i}.png"

        if page >= len(page_numbers):
            print(f"image {filename} exceeds transcribed range")
            continue
        pc = page_numbers[page]
        sents = []
        for s in texts:
            if 'page' not in s:
                continue
            if ('lang' in s and s['lang'] == 'chi' and
                    'type' in s and s['type'] in ['main', 'anno', 'anno2', 'anno3']):
                continue
            pns = s['page'].split('-')
            if pc in pns:
                is_anno = 'type' in s and 'anno' in s['type']
                sents.append((pns, is_anno, s['text']))

        num_border_sents = 0
        for s in sents:
            if len(s[0]) > 1:
                num_border_sents += 1
            if len(s[0]) == 1:
                break

        if num_border_sents > 1:
            print("ERROR: two border sentences", filename, pc)
            print(sents)
        else:
            pages.append({
                'file_name': filename,
                'text': sents,
                'pc': pc,
            })
        page += 1

    return pages


def adaptiveThreshold(image):
    image = cv2.cvtColor(image, cv2.COLOR_RGB2GRAY)
    # image = cv2.medianBlur(image,3)
    image = cv2.GaussianBlur(image, (5, 5), 0)
    image = cv2.adaptiveThreshold(image, 255, cv2.ADAPTIVE_THRESH_MEAN_C, cv2.THRESH_BINARY, 31, 20)
    image = cv2.cvtColor(image, cv2.COLOR_GRAY2RGB)
    return image


def process_page(image, verbose=False, thresholding=False):
    if isinstance(image, str):
        image = cv2.imread(image, cv2.IMREAD_COLOR)
        image = cv2.cvtColor(image, cv2.COLOR_BGR2RGB)
    image_grey = 255 - cv2.cvtColor(image, cv2.COLOR_RGB2GRAY)
    orig_orig_size = (image.shape[1] // 2, image.shape[0] // 2)

    # remove letterbox
    tx, ty, w, h = cv2.boundingRect(cv2.findNonZero(image_grey))
    bbox = ((tx, ty), (tx + w, ty + h))
    image_cropped = image[ty:ty + h, tx:tx + w]
    image_cr = cv2.rotate(image_cropped, cv2.ROTATE_90_COUNTERCLOCKWISE)

    # detect margin
    image_grey = 255 - cv2.cvtColor(image_cr, cv2.COLOR_RGB2GRAY)
    image_grey = cv2.GaussianBlur(image_grey, (7, 7), 0)
    image_resize = cv2.resize(image_grey, (image_grey.shape[1], 1), interpolation=cv2.INTER_AREA)[0]

    x = image_resize[20:-20]
    peaks, properties = find_peaks(x, prominence=20, width=4)

    if verbose:
        plt.plot(x)
        plt.plot(peaks, x[peaks], "x")
        plt.vlines(x=peaks, ymin=x[peaks] - properties["prominences"],
                   ymax=x[peaks], color="C1")
        plt.hlines(y=properties["width_heights"], xmin=properties["left_ips"],
                   xmax=properties["right_ips"], color="C1")
        plt.show()

    ty = max(0, min(peaks) - 50)
    by = min(max(peaks) + 50, image_cr.shape[1])
    image_content = image_cr[:, ty:by]
    bbox = ((bbox[0][0], bbox[0][1] + ty), (bbox[1][0], bbox[0][1] + by))
    image_content = cv2.resize(
        image_content,
        (image_content.shape[1] // 2, image_content.shape[0] // 2),
        interpolation=cv2.INTER_AREA)
    bbox = ((bbox[0][0] // 2, bbox[0][1] // 2), (bbox[1][0] // 2, bbox[1][1] // 2))

    image = cv2.rotate(image_content, cv2.ROTATE_90_CLOCKWISE)

    if thresholding:
        th_image = adaptiveThreshold(image)
        th_image[:, :30] = 255
        th_image[:, -30:] = 255

        image[:, :30] = 255
        image[:, -30:] = 255
        image = np.uint8(th_image * 0.5 + image * 0.5)

    return image, bbox, orig_orig_size


def load_books():
    pages = []
    pages.extend(load_book('월인석보07.json', '월인석보07', 5))
    pages.extend(load_book('월인석보08.json', '월인석보08', 5))
    pages.extend(load_book('석보상절06.json', '석보상절06', 6))

    print(f"{len(pages)}, {len([p for p in pages if len(p['text'][0][0]) == 1])}")

    df = pd.DataFrame(pages)
    return df


HANJA_RE = hanja.build_re()


def cleanup(s):
    s = s.strip().strip('.')
    # s = HANJA_RE.sub('〓', s)
    s = re.sub(r'(?<=[a-zA-Z])\s+(?=[a-zA-Z])', '.', s)
    s = re.sub(r'(?<=[a-zA-Z])\s*(?=' + HANJA_RE.pattern + ')', '.', s)
    s = re.sub(r'(?<=' + HANJA_RE.pattern + r')\s*(?=[a-zA-Z])', '.', s)
    s = re.sub(r'(?<=' + HANJA_RE.pattern + r')\s+(?=' + HANJA_RE.pattern + ')', '', s)
    s = re.sub(r'(?<=' + HANJA_RE.pattern + ')(?=' + HANJA_RE.pattern + ')', '.', s)
    return s.split('.')


def parse_book_text(sentences, cur_page, dgju_dict, verbose=False):
    # find current page
    if verbose:
        print(f"{cur_page=}")

    parsed_spans = []
    last_hanja = None
    for pages, is_anno, sentence in sentences:
        begin = 0
        splits = sentence.split('^')
        split_idx = pages.index(cur_page)
        sentence = splits[split_idx]
        if split_idx > 0:
            last_sent = cleanup(splits[split_idx - 1])
            if HANJA_RE.match(last_sent[-1]):
                last_hanja = last_sent[-1]
                if verbose:
                    print(f"{last_hanja=}")
        for x in re.finditer(r'\[([^]]*)]', sentence):
            match_begin, match_end = x.span(0)
            anno_begin, anno_end = x.span(1)
            parsed_spans.append((pages, is_anno, cleanup(sentence[begin:match_begin])))
            parsed_spans.append((pages, True, cleanup(sentence[anno_begin:anno_end])))
            begin = match_end
        parsed_spans.append((pages, is_anno, cleanup(sentence[begin:])))

    if verbose:
        for pages, is_anno, syllables in parsed_spans:
            print(f"{str(pages):10}\tis_anno={str(is_anno):5}\t{'.'.join(syllables)}")

    page_syllables = []
    for pages, is_anno, syllables in parsed_spans:
        for syllable in syllables:
            page_syllables.append({
                'syllable': syllable,
                'is_anno': is_anno,
            })
            if HANJA_RE.match(syllable):
                page_syllables.append({
                    'syllable': '?',
                    'possibilities': dgju_dict.get(syllable, []),
                    'is_anno': True,
                })

    cand_page_syllables = [page_syllables]
    if last_hanja is not None:
        cand_page_syllables.append([{
            'syllable': '?',
            'possibilities': dgju_dict.get(last_hanja, []),
            'is_anno': True,
        }] + page_syllables)

    if HANJA_RE.match(page_syllables[-1]['syllable']):
        for cand in cand_page_syllables:
            cand_page_syllables.append(cand + [{
                'syllable': '?',
                'possibilities': dgju_dict.get(page_syllables[-1], []),
                'is_anno': True,
            }])

    return cand_page_syllables


def match_syllables(pred_syllables, expected_syllables):
    # Match two strings
    pred_text = '.'.join(pred_syllables)
    expected_text = '.'.join(expected_syllables)
    matches = Levenshtein.matching_blocks(
        Levenshtein.editops(pred_text, expected_text),
        pred_text, expected_text
    )

    match_map = {}
    for match in matches:
        for i in range(match.size):
            match_map[match.a + i] = match.b + i

    # Map text char idx -> syllable idx
    def map_char_to_syllable(syllables):
        result = {}
        offset = 0
        for syll_idx, syllable in enumerate(syllables):
            for i in range(len(syllable)):
                result[offset + i] = syll_idx
            offset += len(syllable) + 1
        return result

    pred_char_to_syll = map_char_to_syllable(pred_syllables)
    gt_char_to_syll = map_char_to_syllable(expected_syllables)

    pred_syll_to_gt_syll = {}  # Map pred syllable idx -> gt syllable idx
    for char_idx, syll_idx in pred_char_to_syll.items():
        if syll_idx not in pred_syll_to_gt_syll:
            pred_syll_to_gt_syll[syll_idx] = []
        gt_char_idx = match_map.get(char_idx)
        if gt_char_idx is not None:
            gt_syll_idx = gt_char_to_syll[gt_char_idx]
            pred_syll_to_gt_syll[syll_idx].append(gt_syll_idx)

    def most_common(lst):
        if len(lst) == 0:
            return None
        data = Counter(lst)
        return data.most_common(1)[0][0]

    pred_syll_to_gt_syll = {
        pred_syll_idx: most_common(gt_syll_idxs)
        for pred_syll_idx, gt_syll_idxs in pred_syll_to_gt_syll.items()
    }

    return pred_syll_to_gt_syll
