import pathlib
from typing import List

import Levenshtein
import numpy as np
from PIL import Image
from tqdm.auto import tqdm

import load_book
import model
import ocr_utils
from syllable_model import SyllableRecognizer


def image_text_align(
        filename: str,
        sentences: List[dict],
        cur_page: str,
        dgju_dict: dict,
        centernet: model.exkp,
        recog: SyllableRecognizer,
        display=None
):
    # Detect page
    orig_image, orig_image_bbox, orig_size = load_book.process_page(filename, thresholding=False)
    pred_syllables, line_infos = ocr_utils.recognize_page(orig_image, centernet, recog, return_line_infos=True)

    pred_bboxes = [item for line in line_infos for item in line['line']]

    # Parse ground truth text
    cand_page_syllables = load_book.parse_book_text(sentences, cur_page, dgju_dict)

    # Construct candidate expected texts
    cand_expected_texts = []
    for cand in cand_page_syllables:
        expected_text = []
        for syllable in cand:
            if load_book.HANJA_RE.match(syllable['syllable']):
                expected_text.append('〓')
            elif syllable['syllable'] == '?' and len(syllable['possibilities']) > 0:
                expected_text.append(syllable['possibilities'][0])
            else:
                expected_text.append(syllable['syllable'])
        cand_expected_texts.append(expected_text)

    if display is not None:
        print("gt  =", '.'.join(cand_expected_texts[0]))
        print("pred=", '.'.join(pred_syllables))

    # Find out which one is correct
    pred_text = '.'.join(pred_syllables)
    leven_dists = [
        Levenshtein.distance(pred_text, '.'.join(cand))
        for cand in cand_expected_texts
    ]
    gt_idx = np.argmin(leven_dists)
    gt_syllables = cand_page_syllables[gt_idx]

    avg_dist = leven_dists[gt_idx] / len(pred_syllables)
    if avg_dist > 2.0:
        print('WARNING: average levenshtein dist > 2.0')
        return False

    # Align text
    expected_text = cand_expected_texts[gt_idx]
    pred_syll_to_gt_syll = load_book.match_syllables(pred_syllables, expected_text)

    # Align text & image
    for pred_syll_idx, (pred, (bbox, _, _, cls)) in enumerate(zip(tqdm(pred_syllables), pred_bboxes)):
        (tlx, tly), (brx, bry) = bbox
        w, h = brx - tlx, bry - tly
        pw, ph = w / 5, h / 5
        tile = orig_image[
            max(0, int(tly - ph)):min(orig_image.shape[0], int(bry + ph)),
            max(0, int(tlx - pw)):min(orig_image.shape[1], int(brx + pw)),
        ]

        # Find corresponding ground truth syllable
        gt_syll_idx = pred_syll_to_gt_syll[pred_syll_idx]
        if gt_syll_idx is None:
            continue
        gt = gt_syllables[gt_syll_idx]

        if load_book.HANJA_RE.match(gt['syllable']):
            possibilities = [gt['syllable']]
        elif 'possibilities' in gt:
            possibilities = gt['possibilities']
        else:
            possibilities = [gt['syllable'] + 'L', gt['syllable'] + 'H', gt['syllable'] + 'R']

        # Skip unknown syllables
        if load_book.HANJA_RE.match(gt['syllable']) or (gt['syllable'] == '?' and len(gt['possibilities']) == 0):
            continue

        # Display syllable
        if display is not None:
            print(pred, possibilities)
            display(Image.fromarray(tile))

        # Predict syllable
        losses = recog.loss([tile] * len(possibilities), possibilities).numpy()
        pred_idx = np.argmin(losses)
        pred_output = possibilities[pred_idx]

        # Save image
        page_id = filename.replace('/', '_').split('.')[0]
        out_path = pathlib.Path(f"real_syllables/{page_id}/{pred_output}_{page_id}_i{pred_syll_idx}.png")
        out_path.parent.mkdir(parents=True, exist_ok=True)
        Image.fromarray(tile).save(out_path)

    return True
