import gradio as gr
import numpy as np
import torch
import io
from PIL import Image
from transformers import PreTrainedModel, VisionEncoderDecoderModel, VisionEncoderDecoderConfig
import cv2
from tqdm.auto import tqdm

import load_book
import utils.hangul
from model import exkp
import syllable_model
import ocr_utils


class OcrModel(PreTrainedModel):
    config_class = VisionEncoderDecoderConfig

    def __init__(self, config):
        super().__init__(config)
        self.centernet = exkp(
            n=5,
            nstack=4,
            dims=[256, 256, 384, 384, 384, 512],
            modules=[2, 2, 2, 2, 2, 4],
            num_classes=4
        )
        self.recog = VisionEncoderDecoderModel(config)

    def forward(self, pixel_values, **kwargs):
        outputs = self.centernet(pixel_values, **kwargs)
        return outputs


def main():
    model = OcrModel.from_pretrained('Chomkwoy/nilkessye')
    if torch.cuda.is_available():
        print("Enabling CUDA")
        model = model.cuda()
    recog = syllable_model.SyllableRecognizer(model.recog)

    def upload_file(file):
        yield (
            [],  # gallery
            "",  # output_textbox
            gr.Textbox(show_label=False, visible=True),  # progress_indicator
        )

        image = Image.open(io.BytesIO(file))
        yield (
            [image],  # gallery
            "",  # output_textbox
            "처리중... 이미지 자르는 중",  # progress_indicator
        )

        image_np = np.array(image)[..., :3]
        generator = recognize_page(
            image_np,
            model, recog,
            return_line_infos=True,
            batch_size=16
        )

        # Crop image
        image = next(generator)
        yield (
            [Image.fromarray(image)],  # gallery
            "",  # output_textbox
            "처리중... 글자 위치 인식 중",  # progress_indicator
        )

        # Get lines
        line_infos = next(generator)
        image = draw_detections(image, line_infos)

        # Read syllables
        num_batches = next(generator)

        yield (
            [Image.fromarray(image)],  # gallery
            "",  # output_textbox
            f"처리중... 글자 읽는 중 (0/{num_batches})",  # progress_indicator
        )

        # Free memory
        i = 0
        while True:
            try:
                pred_syllables = next(generator)
                i += 1
                yield (
                    [Image.fromarray(image)],  # gallery
                    gen_html(pred_syllables, line_infos),  # output_textbox
                    f"처리중... 글자 읽는 중 ({i}/{num_batches})",  # progress_indicator
                )
            except StopIteration:
                break

        yield (
            [Image.fromarray(image)],  # gallery
            gen_html(pred_syllables, line_infos),  # output_textbox
            gr.Textbox(visible=False),  # progress_indicator
        )

    with gr.Blocks() as demo:
        gr.Markdown("""
        # 닐거쎠: 옛한글 글자 인식기

        이미지 파일을 업로드해보세요. 한자는 인식되지 않습니다.
        
        만든사람: ᄎᆞᆷ괴
        """)

        progress_indicator = gr.Textbox(visible=False)

        with gr.Row():
            gallery = gr.Gallery(
                columns=1,
                allow_preview=False,
                object_fit="contain",
                label="보기"
            )

            with gr.Column():
                upload_button = gr.UploadButton(
                    '파일 올리기',
                    type='binary'
                )
                output_textbox = gr.HTML(
                    label="인식 결과",
                    value="여기에 결과가 표시됩니다."
                )

        upload_button.upload(
            fn=upload_file,
            inputs=upload_button,
            outputs=[gallery, output_textbox, progress_indicator]
        )

    demo.queue(max_size=20).launch(server_name='0.0.0.0')


def gen_html(pred_syllables, line_infos):
    output_lines = []
    offset = 0
    for line in line_infos:
        if offset >= len(pred_syllables):
            break
        line_len = len(line['line'])
        cur_line = '.'.join(pred_syllables[offset:offset + line_len])
        cur_line_hangul = utils.hangul.convert_yale_to_hangul(cur_line)
        output_lines.append({
            'is_anno': line['is_anno'],
            'text': cur_line_hangul
        })
        offset += line_len

    output_html = ""
    for line in output_lines:
        if line['is_anno']:
            output_html += f"<span style='letter-spacing: .1rem;'>{line['text']}</span>"
        else:
            output_html += f"<span style='font-size: 1.5em; letter-spacing: .1rem;'>{line['text']}</span>"

    return output_html


def draw_detections(image, line_infos):
    image = image.copy()
    for line_idx, line_info in enumerate(line_infos):
        cv2.rectangle(image,
                      (int(line_info['bbox'][0][0]), int(line_info['bbox'][0][1])),
                      (int(line_info['bbox'][1][0]), int(line_info['bbox'][1][1])),
                      [255, 255, 255], 6)

    for line_idx, line_info in enumerate(line_infos):
        for bbox, center, seq, cls in line_info['line']:
            color = [[160, 158, 255], [212, 56, 13], [107, 255, 171], [255, 205, 66]][int(cls)]
            shapes = image.copy()
            cv2.rectangle(shapes, *bbox, color, cv2.FILLED)
            alpha = 0.75
            image = cv2.addWeighted(image, alpha, shapes, 1 - alpha, 0)
            cv2.rectangle(image, *bbox, color, 2)

    for line_idx, line_info in enumerate(line_infos):
        cv2.putText(
            image, f"{line_idx}",
            (int(line_info['bbox'][0][0]), int(line_info['bbox'][0][1]) + 15),
            cv2.FONT_HERSHEY_SIMPLEX, 0.7, [250, 225, 0], 2
        )
    return image


def recognize_page(orig_image, centernet, syllable_recognizer, return_line_infos=False, batch_size=32):
    orig_image, bbox, orig_size = load_book.process_page(orig_image)
    yield orig_image

    orig_size = (orig_image.shape[1], orig_image.shape[0])
    image = cv2.resize(orig_image, dsize=(512, 512), interpolation=cv2.INTER_AREA)

    image = image.astype(np.float32) / 255. - .5  # to [-.5, +.5] range
    image = image.transpose((2, 0, 1))  # [H, W, C] to [C, H, W]
    image = torch.as_tensor(image)

    # Run object detection
    centernet.eval()
    with torch.no_grad():
        output = centernet(torch.as_tensor(image)[None].to(centernet.device))

    sw, sh = orig_size[0] * 4 / 512, orig_size[1] * 4 / 512

    tiles = ocr_utils.get_pred_detections(
        output, sw=sw, sh=sh,
        threshold=0.3,
        ae_threshold=20.0
    )

    line_infos = ocr_utils.detect_lines(tiles)
    yield line_infos

    yield from recognize_lines(line_infos, orig_image, syllable_recognizer, batch_size=batch_size)


def recognize_lines(line_infos, orig_image, syllable_recognizer, batch_size=32):
    tiles = []
    for line_idx, line_info in enumerate(line_infos):
        for bbox, center, seq, cls in line_info['line']:
            (tlx, tly), (brx, bry) = bbox
            w, h = brx - tlx, bry - tly
            pw, ph = w / 5, h / 5
            tile = orig_image[
                   max(0, int(tly - ph)):min(orig_image.shape[0], int(bry + ph)),
                   max(0, int(tlx - pw)):min(orig_image.shape[1], int(brx + pw)),
                   ]
            tiles.append((tile, bbox, center, seq, cls))

    hangul_tiles = [(i, tile) for i, (tile, _, _, _, cls) in enumerate(tiles) if cls in [0, 2]]

    pred_syllables = ["〓"] * len(tiles)
    batches = list(ocr_utils.batched(hangul_tiles, batch_size))
    yield len(batches)

    for batch in tqdm(batches):
        indices, images = zip(*batch)
        batch_pred_syllables = syllable_recognizer.recognize(images)
        for i, pred_syllable in zip(indices, batch_pred_syllables):
            pred_syllables[i] = pred_syllable
        yield pred_syllables[:i + 1]


if __name__ == "__main__":
    main()
