from transformers import AutoTokenizer, AutoModelForCausalLM
import gradio as gr
import torch

device = "cuda" if torch.cuda.is_available() else "cpu"

tokenizer = AutoTokenizer.from_pretrained("Celestinian/TopicGPT")
model = AutoModelForCausalLM.from_pretrained("Celestinian/TopicGPT")

def generate_text(prompt, temperature, max_size):
    input_ids = tokenizer.encode("#CONTEXT# " + prompt + " #TOPIC#", return_tensors='pt')
    input_ids = input_ids.to(device)
    model.eval()
    model.to(device)

    output_tokens = []
    eos_token_id = tokenizer.encode('#')[0]

    for _ in range(max_size):
        with torch.no_grad():
            outputs = model(input_ids)
        logits = outputs.logits[:, -1, :] / temperature
        next_token = torch.multinomial(torch.softmax(logits, dim=-1), num_samples=1)
        if next_token.item() == eos_token_id:
            break
        input_ids = torch.cat((input_ids, next_token), dim=-1)
        output_tokens.append(next_token.item())

    output = tokenizer.decode(output_tokens)
    clean_output = output.replace('\n', '\n')
    print(prompt + clean_output)
    return clean_output

input_text = gr.inputs.Textbox(lines=5, label="Input Text")
temperature_input = gr.inputs.Slider(minimum=0.01, maximum=2, step=0.01, default=0.01, label="Temperature")
max_size_input = gr.inputs.Slider(minimum=1, maximum=250, step=1, default=30, label="Max Size")
output_text = gr.outputs.Textbox(label="Generated Text")

gr.Interface(generate_text, inputs=[input_text, temperature_input, max_size_input], outputs=output_text).launch()