import argparse
import re
from pprint import pprint

from tokenizers import Tokenizer
from tqdm import trange

from classes import Token
from datamanager import get_data, get_texts, models, set_data, set_data_v3

turbo = False
tokenizer = Tokenizer.from_pretrained("bert-base-uncased")

def process_text(db, db3, text):
    tokens = tokenizer.encode(text).ids
    for i in trange(0, len(tokens), colour="green", unit="tokens"):
        token = tokens[i]
        prev_token = 0 if i == 0 else tokens[i - 1]
        t = Token(token, prev_token, text, tokens, turbo)
        if t not in db:
            db.append(t)
            if prev_token not in db3:
                db3[prev_token] = []
            db3[prev_token].append(t)

def train(model_name):
    db = []
    db3 = {}
    print(f'Rebuilding database for "{model_name}"...')
    k = 0
    texts = get_texts(model_name)
    total_texts = len(texts)
    for text in texts:
        k += 1
        print(f'Processing text {k} of {total_texts}...')
        process_text(db, db3, text)
        
    set_data(model_name, db)
    models[model_name]["db"] = db
    set_data_v3(model_name, db3)
    models[model_name]["db3"] = db3

if __name__ == '__main__':
    parser = argparse.ArgumentParser(
        prog='Text Generator v3',
        description='Generates text from a text file')
    parser.add_argument('-r', '--rebuild', action='extend', nargs="+", type=str)
    parser.add_argument('-l', '--log', action='extend', nargs="+", type=str)
    parser.add_argument('-t', '--turbo', action='store_true')
    args = parser.parse_args()

    if args.rebuild:
        models_to_rebuild = args.rebuild
        if args.rebuild[0] in ('*', 'all'):
            models_to_rebuild = list(models.keys())
        for model in models_to_rebuild:
            if model not in models:
                raise ValueError("Model '%s' not found" % model)
            turbo = args.turbo
            train(model)
    if args.log:
        for model in args.log:
            if model not in models:
                raise ValueError("Model '%s' not found" % model)
            pprint(get_data(model))
